<?php
/**
 * LOX Backup Manager
 *
 * @package LOX_Backup
 */

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Handles backup creation and upload
 */
class LOX_Backup {

    /**
     * Backup directory
     */
    private $backup_dir;

    /**
     * Settings
     */
    private $settings;

    /**
     * API client
     */
    private $api;

    /**
     * Source identifier
     */
    protected $source = 'wordpress';

    /**
     * Component definitions
     */
    protected $components = array(
        'content' => array(
            'name' => 'Content',
            'description' => 'Posts, pages, and media metadata',
            'tables' => array('posts', 'postmeta', 'terms', 'term_taxonomy', 'term_relationships', 'termmeta'),
        ),
        'transactional' => array(
            'name' => 'Users & Comments',
            'description' => 'Users, comments, and session data',
            'tables' => array('users', 'usermeta', 'comments', 'commentmeta'),
        ),
        'files' => array(
            'name' => 'Media Files',
            'description' => 'Uploaded images and documents',
            'paths' => array('uploads'),
        ),
        'config' => array(
            'name' => 'Configuration',
            'description' => 'Settings, plugins, and themes',
            'tables' => array('options'),
            'paths' => array('plugins', 'themes'),
            'files' => array('wp-config.php'),
        ),
    );

    /**
     * Constructor
     */
    public function __construct() {
        $this->settings = get_option('lox_backup_settings', array());
        $this->backup_dir = $this->get_backup_dir();
        $this->api = new LOX_API();
    }

    /**
     * Get component definitions
     */
    public function get_components() {
        return $this->components;
    }

    /**
     * Get source identifier
     */
    public function get_source() {
        return $this->source;
    }

    /**
     * Get backup directory path
     */
    private function get_backup_dir() {
        $upload_dir = wp_upload_dir();
        $backup_dir = $upload_dir['basedir'] . '/lox-backups';

        if (!file_exists($backup_dir)) {
            wp_mkdir_p($backup_dir);

            // Protect directory
            file_put_contents($backup_dir . '/.htaccess', 'deny from all');
            file_put_contents($backup_dir . '/index.php', '<?php // Silence is golden');
        }

        return $backup_dir;
    }

    /**
     * Run full site backup
     *
     * @param array $options Optional settings: frequency, retention_days, immutable_days
     * @return array|WP_Error Result
     */
    public function run_full_backup($options = array()) {
        $start_time = time();
        $timestamp = date('Ymd_His');
        $site_name = sanitize_title(get_bloginfo('name'));
        $backup_name = "wordpress-{$site_name}-{$timestamp}";

        // Extract options with defaults
        $frequency = isset($options['frequency']) ? $options['frequency'] : 'daily';
        $retention_days = isset($options['retention_days']) ? $options['retention_days'] : ($this->settings['retention_days'] ?? null);
        $immutable_days = isset($options['immutable_days']) ? $options['immutable_days'] : ($this->settings['immutable_days'] ?? null);

        // Log start
        $this->log('Starting full backup: ' . $backup_name);

        try {
            // Create temp directory
            $temp_dir = $this->backup_dir . '/temp_' . $timestamp;
            wp_mkdir_p($temp_dir);

            $files_to_backup = array();

            // Backup database
            if (!empty($this->settings['backup_database'])) {
                $db_file = $this->backup_database($temp_dir);
                if (is_wp_error($db_file)) {
                    $this->cleanup($temp_dir);
                    return $db_file;
                }
                $files_to_backup[] = $db_file;
                $this->log('Database backup created');
            }

            // Backup uploads
            if (!empty($this->settings['backup_uploads'])) {
                $uploads_file = $this->backup_directory(WP_CONTENT_DIR . '/uploads', $temp_dir, 'uploads');
                if (!is_wp_error($uploads_file)) {
                    $files_to_backup[] = $uploads_file;
                    $this->log('Uploads backup created');
                }
            }

            // Backup plugins
            if (!empty($this->settings['backup_plugins'])) {
                $plugins_file = $this->backup_directory(WP_PLUGIN_DIR, $temp_dir, 'plugins');
                if (!is_wp_error($plugins_file)) {
                    $files_to_backup[] = $plugins_file;
                    $this->log('Plugins backup created');
                }
            }

            // Backup themes
            if (!empty($this->settings['backup_themes'])) {
                $themes_file = $this->backup_directory(get_theme_root(), $temp_dir, 'themes');
                if (!is_wp_error($themes_file)) {
                    $files_to_backup[] = $themes_file;
                    $this->log('Themes backup created');
                }
            }

            // Backup wp-config.php
            $config_file = $this->backup_config($temp_dir);
            if (!is_wp_error($config_file)) {
                $files_to_backup[] = $config_file;
            }

            // Create combined archive
            $archive_path = $this->backup_dir . '/' . $backup_name . '.tar.gz';
            $this->create_archive($temp_dir, $archive_path);
            $this->log('Archive created: ' . $this->format_bytes(filesize($archive_path)));

            // Upload to LOX
            $this->log("Uploading to LOX (frequency: {$frequency})...");
            $result = $this->api->upload_backup($archive_path, array(
                'name' => $backup_name,
                'tags' => ($this->settings['tags'] ?? 'wordpress,automated') . ",{$frequency}",
                'retention_days' => $retention_days,  // null = use account frequency-based default
                'immutable_days' => $immutable_days,
                'frequency' => $frequency,
                'description' => sprintf(
                    'WordPress backup from %s (%s)',
                    home_url(),
                    get_bloginfo('name')
                ),
                'source' => $this->source,
                'component' => 'full',
            ));

            if (is_wp_error($result)) {
                $this->cleanup($temp_dir, $archive_path);
                $this->update_status('failed', $result->get_error_message());
                return $result;
            }

            $this->log('Upload complete, waiting for processing...');

            // Wait for completion
            $backup = $this->api->wait_for_completion($result['uuid'], 3600);

            // Cleanup
            $this->cleanup($temp_dir, $archive_path);

            if (is_wp_error($backup)) {
                $this->update_status('failed', $backup->get_error_message());
                return $backup;
            }

            $duration = time() - $start_time;
            $this->log(sprintf('Backup completed in %d seconds. UUID: %s', $duration, $backup['uuid']));
            $this->update_status('completed', null, $backup);

            return $backup;

        } catch (Exception $e) {
            $this->log('Backup failed: ' . $e->getMessage(), 'error');
            $this->update_status('failed', $e->getMessage());
            return new WP_Error('backup_failed', $e->getMessage());
        }
    }

    /**
     * Run component backup
     *
     * @param string $component Component name (content, transactional, files, config)
     * @param array $options Optional settings: frequency, retention_days, immutable_days
     * @return array|WP_Error Result
     */
    public function run_component_backup($component, $options = array()) {
        if (!isset($this->components[$component])) {
            return new WP_Error('invalid_component', __('Invalid backup component', 'lox-backup'));
        }

        $start_time = time();
        $timestamp = date('Ymd_His');
        $site_name = sanitize_title(get_bloginfo('name'));
        $backup_name = "{$this->source}-{$site_name}-{$component}-{$timestamp}";
        $component_def = $this->components[$component];

        $this->log("Starting {$component} backup: {$backup_name}");

        try {
            // Create temp directory
            $temp_dir = $this->backup_dir . '/temp_' . $component . '_' . $timestamp;
            wp_mkdir_p($temp_dir);

            // Backup tables if defined
            if (!empty($component_def['tables'])) {
                $db_file = $this->backup_component_tables($temp_dir, $component_def['tables']);
                if (is_wp_error($db_file)) {
                    $this->cleanup($temp_dir);
                    return $db_file;
                }
                $this->log("Database tables backed up for {$component}");
            }

            // Backup paths if defined
            if (!empty($component_def['paths'])) {
                foreach ($component_def['paths'] as $path) {
                    $full_path = WP_CONTENT_DIR . '/' . $path;
                    if (is_dir($full_path)) {
                        $archive = $this->backup_directory($full_path, $temp_dir, $path);
                        if (!is_wp_error($archive)) {
                            $this->log("Directory {$path} backed up");
                        }
                    }
                }
            }

            // Backup specific files if defined
            if (!empty($component_def['files'])) {
                foreach ($component_def['files'] as $file) {
                    if ($file === 'wp-config.php') {
                        $this->backup_config($temp_dir);
                    }
                }
            }

            // Create archive
            $archive_path = $this->backup_dir . '/' . $backup_name . '.tar.gz';
            $this->create_archive($temp_dir, $archive_path);
            $this->log("Archive created: " . $this->format_bytes(filesize($archive_path)));

            // Upload to LOX - use frequency from options or default to daily
            $frequency = isset($options['frequency']) ? $options['frequency'] : 'daily';
            $retention_days = isset($options['retention_days']) ? $options['retention_days'] : ($this->settings['retention_days'] ?? null);
            $immutable_days = isset($options['immutable_days']) ? $options['immutable_days'] : ($this->settings['immutable_days'] ?? null);

            $this->log("Uploading {$component} backup to LOX (frequency: {$frequency})...");
            $result = $this->api->upload_backup($archive_path, array(
                'name' => $backup_name,
                'tags' => "{$this->source},{$component},automated,{$frequency}",
                'retention_days' => $retention_days,  // null = use account frequency-based default
                'immutable_days' => $immutable_days,
                'frequency' => $frequency,
                'description' => sprintf(
                    '%s %s backup from %s (%s)',
                    ucfirst($this->source),
                    $component_def['name'],
                    home_url(),
                    get_bloginfo('name')
                ),
                'source' => $this->source,
                'component' => $component,
            ));

            if (is_wp_error($result)) {
                $this->cleanup($temp_dir, $archive_path);
                $this->update_component_status($component, 'failed', $result->get_error_message());
                return $result;
            }

            // Wait for completion
            $backup = $this->api->wait_for_completion($result['uuid'], 1800);

            // Cleanup
            $this->cleanup($temp_dir, $archive_path);

            if (is_wp_error($backup)) {
                $this->update_component_status($component, 'failed', $backup->get_error_message());
                return $backup;
            }

            $duration = time() - $start_time;
            $this->log(sprintf('%s backup completed in %d seconds. UUID: %s', ucfirst($component), $duration, $backup['uuid']));
            $this->update_component_status($component, 'completed', null, $backup);

            return $backup;

        } catch (Exception $e) {
            $this->log("Component backup failed: " . $e->getMessage(), 'error');
            $this->update_component_status($component, 'failed', $e->getMessage());
            return new WP_Error('backup_failed', $e->getMessage());
        }
    }

    /**
     * Backup specific database tables
     *
     * @param string $temp_dir Temp directory
     * @param array $tables Table names (without prefix)
     * @return string|WP_Error SQL file path
     */
    private function backup_component_tables($temp_dir, $tables) {
        global $wpdb;

        $sql_file = $temp_dir . '/database.sql';
        $handle = fopen($sql_file, 'w');

        if (!$handle) {
            return new WP_Error('db_backup_failed', __('Failed to create database backup file', 'lox-backup'));
        }

        // Write header
        fwrite($handle, "-- LOX WordPress Component Backup\n");
        fwrite($handle, "-- Generated: " . date('Y-m-d H:i:s') . "\n");
        fwrite($handle, "-- Site: " . home_url() . "\n");
        fwrite($handle, "-- Tables: " . implode(', ', $tables) . "\n\n");
        fwrite($handle, "SET NAMES utf8mb4;\n");
        fwrite($handle, "SET FOREIGN_KEY_CHECKS = 0;\n\n");

        foreach ($tables as $table_suffix) {
            $table_name = $wpdb->prefix . $table_suffix;

            // Check if table exists
            $exists = $wpdb->get_var("SHOW TABLES LIKE '{$table_name}'");
            if (!$exists) {
                continue;
            }

            // Table structure
            $create = $wpdb->get_row("SHOW CREATE TABLE `{$table_name}`", ARRAY_N);
            fwrite($handle, "DROP TABLE IF EXISTS `{$table_name}`;\n");
            fwrite($handle, $create[1] . ";\n\n");

            // Table data
            $rows = $wpdb->get_results("SELECT * FROM `{$table_name}`", ARRAY_A);

            if (!empty($rows)) {
                $columns = array_keys($rows[0]);
                $column_list = '`' . implode('`, `', $columns) . '`';

                foreach (array_chunk($rows, 100) as $chunk) {
                    $values = array();
                    foreach ($chunk as $row) {
                        $escaped = array_map(function($v) use ($wpdb) {
                            return $v === null ? 'NULL' : "'" . $wpdb->_real_escape($v) . "'";
                        }, array_values($row));
                        $values[] = '(' . implode(', ', $escaped) . ')';
                    }
                    fwrite($handle, "INSERT INTO `{$table_name}` ({$column_list}) VALUES\n" . implode(",\n", $values) . ";\n");
                }
                fwrite($handle, "\n");
            }
        }

        fwrite($handle, "SET FOREIGN_KEY_CHECKS = 1;\n");
        fclose($handle);

        return $sql_file;
    }

    /**
     * Update component backup status
     *
     * @param string $component Component name
     * @param string $status Status
     * @param string|null $message Error message
     * @param array|null $backup Backup data
     */
    private function update_component_status($component, $status, $message = null, $backup = null) {
        $component_status = get_option('lox_backup_component_status', array());

        $component_status[$component] = array(
            'last_backup' => current_time('mysql'),
            'status' => $status,
            'message' => $message,
            'uuid' => $backup ? $backup['uuid'] : null,
            'size' => $backup ? $backup['size_bytes'] : null,
        );

        update_option('lox_backup_component_status', $component_status);
    }

    /**
     * Get component backup status
     *
     * @return array Component statuses
     */
    public function get_component_status() {
        return get_option('lox_backup_component_status', array());
    }

    /**
     * Backup database
     *
     * @param string $temp_dir Temp directory
     * @return string|WP_Error SQL file path
     */
    private function backup_database($temp_dir) {
        global $wpdb;

        $sql_file = $temp_dir . '/database.sql';
        $handle = fopen($sql_file, 'w');

        if (!$handle) {
            return new WP_Error('db_backup_failed', __('Failed to create database backup file', 'lox-backup'));
        }

        // Write header
        fwrite($handle, "-- LOX WordPress Backup\n");
        fwrite($handle, "-- Generated: " . date('Y-m-d H:i:s') . "\n");
        fwrite($handle, "-- Site: " . home_url() . "\n\n");
        fwrite($handle, "SET NAMES utf8mb4;\n");
        fwrite($handle, "SET FOREIGN_KEY_CHECKS = 0;\n\n");

        // Get tables
        $tables = $wpdb->get_results("SHOW TABLES", ARRAY_N);

        foreach ($tables as $table) {
            $table_name = $table[0];

            // Skip non-WordPress tables if prefix is set
            if ($wpdb->prefix && strpos($table_name, $wpdb->prefix) !== 0) {
                continue;
            }

            // Table structure
            $create = $wpdb->get_row("SHOW CREATE TABLE `{$table_name}`", ARRAY_N);
            fwrite($handle, "DROP TABLE IF EXISTS `{$table_name}`;\n");
            fwrite($handle, $create[1] . ";\n\n");

            // Table data
            $rows = $wpdb->get_results("SELECT * FROM `{$table_name}`", ARRAY_A);

            if (!empty($rows)) {
                $columns = array_keys($rows[0]);
                $column_list = '`' . implode('`, `', $columns) . '`';

                foreach (array_chunk($rows, 100) as $chunk) {
                    $values = array();
                    foreach ($chunk as $row) {
                        $escaped = array_map(function($v) use ($wpdb) {
                            return $v === null ? 'NULL' : "'" . $wpdb->_real_escape($v) . "'";
                        }, array_values($row));
                        $values[] = '(' . implode(', ', $escaped) . ')';
                    }
                    fwrite($handle, "INSERT INTO `{$table_name}` ({$column_list}) VALUES\n" . implode(",\n", $values) . ";\n");
                }
                fwrite($handle, "\n");
            }
        }

        fwrite($handle, "SET FOREIGN_KEY_CHECKS = 1;\n");
        fclose($handle);

        return $sql_file;
    }

    /**
     * Backup a directory
     *
     * @param string $source Source directory
     * @param string $temp_dir Temp directory
     * @param string $name Archive name
     * @return string|WP_Error Archive path
     */
    private function backup_directory($source, $temp_dir, $name) {
        if (!is_dir($source)) {
            return new WP_Error('dir_not_found', sprintf(__('Directory not found: %s', 'lox-backup'), $source));
        }

        $archive_path = $temp_dir . '/' . $name . '.tar';

        // Remove existing archive from PHAR registry if exists
        if (file_exists($archive_path)) {
            try {
                Phar::unlinkArchive($archive_path);
            } catch (Exception $e) {
                @unlink($archive_path);
            }
        }

        try {
            $phar = new PharData($archive_path);
            $phar->buildFromDirectory($source);
            unset($phar);
            return $archive_path;
        } catch (Exception $e) {
            return new WP_Error('archive_failed', $e->getMessage());
        }
    }

    /**
     * Backup wp-config.php
     *
     * @param string $temp_dir Temp directory
     * @return string|WP_Error File path
     */
    private function backup_config($temp_dir) {
        $config_path = ABSPATH . 'wp-config.php';

        if (!file_exists($config_path)) {
            $config_path = dirname(ABSPATH) . '/wp-config.php';
        }

        if (!file_exists($config_path)) {
            return new WP_Error('config_not_found', __('wp-config.php not found', 'lox-backup'));
        }

        $dest_path = $temp_dir . '/wp-config.php';
        copy($config_path, $dest_path);

        return $dest_path;
    }

    /**
     * Create final archive
     *
     * @param string $source_dir Source directory
     * @param string $archive_path Archive path
     */
    private function create_archive($source_dir, $archive_path) {
        $tar_path = str_replace('.tar.gz', '.tar', $archive_path);

        // Remove existing archives from filesystem and PHAR registry
        if (file_exists($archive_path)) {
            try {
                Phar::unlinkArchive($archive_path);
            } catch (Exception $e) {
                @unlink($archive_path);
            }
        }
        if (file_exists($tar_path)) {
            try {
                Phar::unlinkArchive($tar_path);
            } catch (Exception $e) {
                @unlink($tar_path);
            }
        }

        // Create new archive
        $phar = new PharData($tar_path);
        $phar->buildFromDirectory($source_dir);
        $phar->compress(Phar::GZ);

        // Cleanup: unset reference and remove uncompressed tar
        unset($phar);
        if (file_exists($tar_path)) {
            try {
                Phar::unlinkArchive($tar_path);
            } catch (Exception $e) {
                @unlink($tar_path);
            }
        }
    }

    /**
     * Cleanup temporary files
     *
     * @param string $temp_dir Temp directory
     * @param string $archive Archive file
     */
    private function cleanup($temp_dir, $archive = null) {
        // Remove temp directory
        if (is_dir($temp_dir)) {
            $this->recursive_delete($temp_dir);
        }

        // Remove archive
        if ($archive && file_exists($archive)) {
            unlink($archive);
        }
    }

    /**
     * Recursively delete directory
     *
     * @param string $dir Directory path
     */
    private function recursive_delete($dir) {
        if (is_dir($dir)) {
            $objects = scandir($dir);
            foreach ($objects as $object) {
                if ($object != '.' && $object != '..') {
                    if (is_dir($dir . '/' . $object)) {
                        $this->recursive_delete($dir . '/' . $object);
                    } else {
                        unlink($dir . '/' . $object);
                    }
                }
            }
            rmdir($dir);
        }
    }

    /**
     * Update backup status
     *
     * @param string $status Status
     * @param string|null $message Error message
     * @param array|null $backup Backup data
     */
    private function update_status($status, $message = null, $backup = null) {
        $this->settings['last_backup'] = current_time('mysql');
        $this->settings['last_backup_status'] = $status;
        $this->settings['last_backup_message'] = $message;

        if ($backup) {
            $this->settings['last_backup_uuid'] = $backup['uuid'];
            $this->settings['last_backup_size'] = $backup['size_bytes'];
        }

        update_option('lox_backup_settings', $this->settings);
    }

    /**
     * Log message
     *
     * @param string $message Message
     * @param string $level Log level
     */
    private function log($message, $level = 'info') {
        if (defined('WP_DEBUG') && WP_DEBUG) {
            error_log(sprintf('[LOX Backup] [%s] %s', strtoupper($level), $message));
        }
    }

    /**
     * Format bytes to human readable
     *
     * @param int $bytes Bytes
     * @return string Formatted string
     */
    private function format_bytes($bytes) {
        $units = array('B', 'KB', 'MB', 'GB', 'TB');
        $bytes = max($bytes, 0);
        $pow = floor(($bytes ? log($bytes) : 0) / log(1024));
        $pow = min($pow, count($units) - 1);
        $bytes /= pow(1024, $pow);
        return round($bytes, 2) . ' ' . $units[$pow];
    }

    /**
     * Run custom backup with selected elements
     *
     * @param array $elements Array of elements to backup: database, uploads, plugins, themes, config
     * @param string $profile_name Optional profile name for labeling
     * @param array $options Optional settings: frequency, retention_days, immutable_days
     * @return array|WP_Error Result
     */
    public function run_custom_backup($elements, $profile_name = '', $options = array()) {
        if (empty($elements)) {
            return new WP_Error('no_elements', __('No backup elements selected', 'lox-backup'));
        }

        $start_time = time();
        $timestamp = date('Ymd_His');
        $site_name = sanitize_title(get_bloginfo('name'));

        // Extract options with defaults
        $frequency = isset($options['frequency']) ? $options['frequency'] : 'daily';
        $retention_days = isset($options['retention_days']) ? $options['retention_days'] : ($this->settings['retention_days'] ?? null);
        $immutable_days = isset($options['immutable_days']) ? $options['immutable_days'] : ($this->settings['immutable_days'] ?? null);
        $profile_uuid = isset($options['profile_uuid']) ? $options['profile_uuid'] : null;

        // Create descriptive name
        $elements_slug = implode('-', array_map('sanitize_title', $elements));
        if ($profile_name) {
            $backup_name = "wordpress-{$site_name}-" . sanitize_title($profile_name) . "-{$timestamp}";
        } else {
            $backup_name = "wordpress-{$site_name}-custom-{$elements_slug}-{$timestamp}";
        }

        $this->log('Starting custom backup: ' . $backup_name . ' [' . implode(', ', $elements) . '] (frequency: ' . $frequency . ')');

        try {
            // Create temp directory
            $temp_dir = $this->backup_dir . '/temp_custom_' . $timestamp;
            wp_mkdir_p($temp_dir);

            $files_to_backup = array();
            $backed_up_elements = array();

            // Backup database if selected
            if (in_array('database', $elements)) {
                $db_file = $this->backup_database($temp_dir);
                if (is_wp_error($db_file)) {
                    $this->cleanup($temp_dir);
                    return $db_file;
                }
                $files_to_backup[] = $db_file;
                $backed_up_elements[] = 'database';
                $this->log('Database backup created');
            }

            // Backup uploads if selected
            if (in_array('uploads', $elements)) {
                $uploads_file = $this->backup_directory(WP_CONTENT_DIR . '/uploads', $temp_dir, 'uploads');
                if (!is_wp_error($uploads_file)) {
                    $files_to_backup[] = $uploads_file;
                    $backed_up_elements[] = 'uploads';
                    $this->log('Uploads backup created');
                }
            }

            // Backup plugins if selected
            if (in_array('plugins', $elements)) {
                $plugins_file = $this->backup_directory(WP_PLUGIN_DIR, $temp_dir, 'plugins');
                if (!is_wp_error($plugins_file)) {
                    $files_to_backup[] = $plugins_file;
                    $backed_up_elements[] = 'plugins';
                    $this->log('Plugins backup created');
                }
            }

            // Backup themes if selected
            if (in_array('themes', $elements)) {
                $themes_file = $this->backup_directory(get_theme_root(), $temp_dir, 'themes');
                if (!is_wp_error($themes_file)) {
                    $files_to_backup[] = $themes_file;
                    $backed_up_elements[] = 'themes';
                    $this->log('Themes backup created');
                }
            }

            // Backup wp-config if selected
            if (in_array('config', $elements)) {
                $config_file = $this->backup_config($temp_dir);
                if (!is_wp_error($config_file)) {
                    $files_to_backup[] = $config_file;
                    $backed_up_elements[] = 'config';
                    $this->log('Config backup created');
                }
            }

            if (empty($files_to_backup)) {
                $this->cleanup($temp_dir);
                return new WP_Error('backup_empty', __('No files were backed up', 'lox-backup'));
            }

            // Create combined archive
            $archive_path = $this->backup_dir . '/' . $backup_name . '.tar.gz';
            $this->create_archive($temp_dir, $archive_path);
            $this->log('Archive created: ' . $this->format_bytes(filesize($archive_path)));

            // Build tags
            $tags = array('wordpress', 'custom');
            $tags = array_merge($tags, $backed_up_elements);
            if ($profile_name) {
                $tags[] = 'profile:' . sanitize_title($profile_name);
            }

            // Upload to LOX
            $tags[] = $frequency;  // Add frequency to tags
            $this->log("Uploading custom backup to LOX (frequency: {$frequency}, profile: " . ($profile_uuid ?: 'none') . ")...");
            $result = $this->api->upload_backup($archive_path, array(
                'name' => $backup_name,
                'tags' => implode(',', $tags),
                'retention_days' => $retention_days,  // null = use account frequency-based default
                'immutable_days' => $immutable_days,
                'frequency' => $frequency,
                'description' => sprintf(
                    'WordPress custom backup (%s) from %s (%s)%s',
                    implode(', ', $backed_up_elements),
                    home_url(),
                    get_bloginfo('name'),
                    $profile_name ? " - Profile: {$profile_name}" : ''
                ),
                'source' => $this->source,
                'component' => 'custom',
                'profile_uuid' => $profile_uuid,
            ));

            if (is_wp_error($result)) {
                $this->cleanup($temp_dir, $archive_path);
                $this->update_status('failed', $result->get_error_message());
                return $result;
            }

            $this->log('Upload complete, waiting for processing...');

            // Wait for completion
            $backup = $this->api->wait_for_completion($result['uuid'], 3600);

            // Cleanup
            $this->cleanup($temp_dir, $archive_path);

            if (is_wp_error($backup)) {
                $this->update_status('failed', $backup->get_error_message());
                return $backup;
            }

            $duration = time() - $start_time;
            $this->log(sprintf('Custom backup completed in %d seconds. UUID: %s', $duration, $backup['uuid']));
            $this->update_status('completed', null, $backup);

            return $backup;

        } catch (Exception $e) {
            $this->log('Custom backup failed: ' . $e->getMessage(), 'error');
            $this->update_status('failed', $e->getMessage());
            return new WP_Error('backup_failed', $e->getMessage());
        }
    }

    /**
     * Run a saved backup profile
     *
     * @param string $profile_id Profile ID
     * @param array $override_options Optional settings to override profile defaults
     * @return array|WP_Error Result
     */
    public function run_profile_backup($profile_id, $override_options = array()) {
        $profiles = get_option('lox_backup_profiles', array());

        if (!isset($profiles[$profile_id])) {
            return new WP_Error('profile_not_found', __('Backup profile not found', 'lox-backup'));
        }

        $profile = $profiles[$profile_id];

        // Build options from profile settings and overrides
        $options = array(
            'frequency' => isset($override_options['frequency']) ? $override_options['frequency'] : ($profile['schedule'] ?? 'daily'),
            'retention_days' => isset($override_options['retention_days']) ? $override_options['retention_days'] : ($profile['retention_days'] ?? null),
            'immutable_days' => isset($override_options['immutable_days']) ? $override_options['immutable_days'] : ($profile['immutable_days'] ?? null),
            'profile_uuid' => $profile['remote_uuid'] ?? null,
        );

        return $this->run_custom_backup($profile['elements'], $profile['name'], $options);
    }

    /**
     * Get available backup elements
     *
     * @return array Elements with labels and descriptions
     */
    public static function get_backup_elements() {
        return array(
            'database' => array(
                'label' => __('Database', 'lox-backup'),
                'description' => __('All WordPress database tables (posts, users, settings, etc.)', 'lox-backup'),
                'icon' => 'dashicons-database',
                'size_estimate' => 'small',
            ),
            'uploads' => array(
                'label' => __('Media Uploads', 'lox-backup'),
                'description' => __('Images, videos, and documents from media library', 'lox-backup'),
                'icon' => 'dashicons-format-image',
                'size_estimate' => 'large',
            ),
            'plugins' => array(
                'label' => __('Plugins', 'lox-backup'),
                'description' => __('All installed plugins', 'lox-backup'),
                'icon' => 'dashicons-admin-plugins',
                'size_estimate' => 'medium',
            ),
            'themes' => array(
                'label' => __('Themes', 'lox-backup'),
                'description' => __('All installed themes', 'lox-backup'),
                'icon' => 'dashicons-admin-appearance',
                'size_estimate' => 'medium',
            ),
            'config' => array(
                'label' => __('Configuration', 'lox-backup'),
                'description' => __('wp-config.php file', 'lox-backup'),
                'icon' => 'dashicons-admin-settings',
                'size_estimate' => 'tiny',
            ),
        );
    }
}
