<?php
/**
 * LOX API Client
 *
 * @package LOX_Backup
 */

if (!defined('ABSPATH')) {
    exit;
}

/**
 * API client for LOX Backup Service
 */
class LOX_API {

    /**
     * API key
     */
    private $api_key;

    /**
     * Base URL
     */
    private $base_url;

    /**
     * Timeout in seconds
     */
    private $timeout = 300;

    /**
     * Constructor
     */
    public function __construct($api_key = null, $base_url = null) {
        $settings = get_option('lox_backup_settings', array());

        $this->api_key = $api_key ?: (isset($settings['api_key']) ? $settings['api_key'] : '');
        $this->base_url = $base_url ?: (isset($settings['base_url']) ? $settings['base_url'] : 'https://backlox.com/api');
    }

    /**
     * Test API connection
     *
     * @return array|WP_Error
     */
    public function test_connection() {
        return $this->request('GET', '/v1/tenants/me');
    }

    /**
     * Get tenant quota
     *
     * @return array|WP_Error
     */
    public function get_quota() {
        return $this->request('GET', '/v1/tenants/usage');
    }

    /**
     * Upload backup file
     *
     * @param string $file_path Path to backup file
     * @param array $options Backup options
     * @return array|WP_Error
     */
    public function upload_backup($file_path, $options = array()) {
        if (!file_exists($file_path)) {
            return new WP_Error('lox_upload_file_not_found', __('Backup file not found', 'lox-backup'));
        }

        $defaults = array(
            'name' => basename($file_path),
            'tags' => 'wordpress',
            'retention_days' => null,  // null = use frequency-based default from account settings
            'immutable_days' => null,  // null = use frequency-based default from account settings
            'frequency' => null,       // hourly, daily, weekly, monthly, yearly - determines retention policy
            'description' => sprintf('WordPress backup from %s', home_url()),
            'source' => 'wordpress',
            'component' => null,
            'source_identifier' => $this->get_site_identifier(),
            'profile_uuid' => null,  // Optional: assign to specific profile
            'extra_metadata' => null,  // Optional: site metadata (wp_version, etc.)
        );

        $options = wp_parse_args($options, $defaults);

        // Auto-add site metadata if not provided
        if ($options['extra_metadata'] === null) {
            $options['extra_metadata'] = json_encode($this->get_site_metadata());
        } elseif (is_array($options['extra_metadata'])) {
            $options['extra_metadata'] = json_encode($options['extra_metadata']);
        }

        // Prepare multipart request
        $boundary = wp_generate_password(24, false);

        $body = '';

        // Add file
        $body .= "--{$boundary}\r\n";
        $body .= 'Content-Disposition: form-data; name="file"; filename="' . basename($file_path) . '"' . "\r\n";
        $body .= "Content-Type: application/gzip\r\n\r\n";
        $body .= file_get_contents($file_path);
        $body .= "\r\n";

        // Add form fields
        $fields = array('name', 'tags', 'retention_days', 'immutable_days', 'frequency', 'description', 'source', 'component', 'source_identifier', 'profile_uuid', 'extra_metadata');
        foreach ($fields as $field) {
            if (isset($options[$field]) && $options[$field] !== null) {
                $body .= "--{$boundary}\r\n";
                $body .= 'Content-Disposition: form-data; name="' . $field . '"' . "\r\n\r\n";
                $body .= $options[$field] . "\r\n";
            }
        }

        $body .= "--{$boundary}--\r\n";

        $response = wp_remote_post(
            $this->base_url . '/v1/backups',
            array(
                'timeout' => $this->timeout,
                'headers' => array(
                    'X-API-Key' => $this->api_key,
                    'Content-Type' => 'multipart/form-data; boundary=' . $boundary,
                    'User-Agent' => 'LOX-WordPress/' . LOX_BACKUP_VERSION,
                ),
                'body' => $body,
            )
        );

        return $this->handle_response($response);
    }

    /**
     * Get site metadata for backup profile
     *
     * @return array Site metadata
     */
    public function get_site_metadata() {
        global $wp_version;

        return array(
            'site_url' => get_site_url(),
            'site_name' => get_bloginfo('name'),
            'wp_version' => $wp_version,
            'php_version' => phpversion(),
            'is_multisite' => is_multisite(),
            'theme' => wp_get_theme()->get('Name'),
            'plugin_count' => count(get_plugins()),
            'locale' => get_locale(),
        );
    }

    /**
     * Get backup status
     *
     * @param string $uuid Backup UUID
     * @return array|WP_Error
     */
    public function get_backup_status($uuid) {
        return $this->request('GET', '/v1/backups/' . $uuid);
    }

    /**
     * Wait for backup completion
     *
     * @param string $uuid Backup UUID
     * @param int $timeout Timeout in seconds
     * @return array|WP_Error
     */
    public function wait_for_completion($uuid, $timeout = 3600) {
        $start_time = time();
        $poll_interval = 5;

        while (true) {
            $backup = $this->get_backup_status($uuid);

            if (is_wp_error($backup)) {
                return $backup;
            }

            if ($backup['status'] === 'completed') {
                return $backup;
            }

            if ($backup['status'] === 'failed') {
                return new WP_Error(
                    'lox_backup_failed',
                    sprintf(__('Backup failed: %s', 'lox-backup'), $backup['status_message'] ?? 'Unknown error')
                );
            }

            if ($backup['status'] === 'quarantine') {
                return new WP_Error(
                    'lox_backup_quarantine',
                    __('Backup quarantined: Potential malware detected', 'lox-backup')
                );
            }

            if ((time() - $start_time) >= $timeout) {
                return new WP_Error('lox_backup_timeout', __('Timeout waiting for backup completion', 'lox-backup'));
            }

            sleep($poll_interval);
        }
    }

    /**
     * List backups
     *
     * @param array $params Query parameters
     * @return array|WP_Error
     */
    public function list_backups($params = array()) {
        return $this->request('GET', '/v1/backups', $params);
    }

    /**
     * Get latest backups by component
     *
     * @param string $source Source identifier (wordpress, woocommerce)
     * @param string $source_identifier Optional site identifier
     * @return array|WP_Error
     */
    public function get_latest_backups($source = 'wordpress', $source_identifier = null) {
        $params = array('source' => $source);
        if ($source_identifier === null) {
            $source_identifier = $this->get_site_identifier();
        }
        if ($source_identifier) {
            $params['source_identifier'] = $source_identifier;
        }
        return $this->request('GET', '/v1/backups/latest', $params);
    }

    /**
     * Get backup profiles
     *
     * @param array $params Query parameters (source, source_identifier, is_active)
     * @return array|WP_Error
     */
    public function get_profiles($params = array()) {
        $defaults = array(
            'source' => 'wordpress',
            'source_identifier' => $this->get_site_identifier(),
        );
        $params = wp_parse_args($params, $defaults);
        return $this->request('GET', '/v1/backup-profiles/', $params);  // Trailing slash required
    }

    /**
     * Get a specific backup profile with its backups
     *
     * @param string $uuid Profile UUID
     * @param int $limit Number of backups to include
     * @return array|WP_Error
     */
    public function get_profile($uuid, $limit = 10) {
        return $this->request('GET', '/v1/backup-profiles/' . $uuid, array('limit' => $limit));
    }

    /**
     * Get all versions (backups) for a profile
     *
     * @param string $uuid Profile UUID
     * @param int $page Page number
     * @param int $per_page Items per page
     * @return array|WP_Error
     */
    public function get_profile_versions($uuid, $page = 1, $per_page = 20) {
        return $this->request('GET', '/v1/backup-profiles/' . $uuid . '/versions', array(
            'page' => $page,
            'per_page' => $per_page,
        ));
    }

    /**
     * Create a custom backup profile
     *
     * @param string $name Profile name
     * @param array $options Profile options
     * @return array|WP_Error
     */
    public function create_profile($name, $options = array()) {
        $defaults = array(
            'name' => $name,
            'source' => 'wordpress',
            'source_identifier' => $this->get_site_identifier(),
            'is_custom' => true,
        );
        $data = wp_parse_args($options, $defaults);
        return $this->request('POST', '/v1/backup-profiles', $data);
    }

    /**
     * Update a backup profile
     *
     * @param string $uuid Profile UUID
     * @param array $data Update data
     * @return array|WP_Error
     */
    public function update_profile($uuid, $data) {
        return $this->request('PATCH', '/v1/backup-profiles/' . $uuid, $data);
    }

    /**
     * Generate a unique identifier for this WordPress installation
     *
     * Uses a hash of the site URL to create a consistent, privacy-friendly identifier.
     *
     * @return string Site identifier
     */
    public function get_site_identifier() {
        // Check if we have a stored identifier
        $stored_id = get_option('lox_backup_site_identifier');
        if ($stored_id) {
            return $stored_id;
        }

        // Generate identifier based on site URL
        $site_url = get_site_url();
        $parsed = parse_url($site_url);

        // Create a readable identifier: domain + path hash
        $domain = isset($parsed['host']) ? $parsed['host'] : 'localhost';
        $path = isset($parsed['path']) ? $parsed['path'] : '';

        // For multisite or subdirectory installs, include the path
        if (!empty($path) && $path !== '/') {
            $identifier = $domain . str_replace('/', '-', rtrim($path, '/'));
        } else {
            $identifier = $domain;
        }

        // Add a short hash for uniqueness (in case of domain changes)
        $hash = substr(md5($site_url . AUTH_KEY), 0, 8);
        $identifier = $identifier . '-' . $hash;

        // Store for consistency
        update_option('lox_backup_site_identifier', $identifier);

        return $identifier;
    }

    /**
     * Request backup restore
     *
     * @param string $uuid Backup UUID
     * @return array|WP_Error
     */
    public function request_restore($uuid) {
        return $this->request('POST', '/v1/backups/' . $uuid . '/restore', array(
            'priority' => 'normal'
        ));
    }

    /**
     * Get download URL for backup
     *
     * @param string $uuid Backup UUID
     * @return array|WP_Error
     */
    public function get_download_url($uuid) {
        return $this->request('GET', '/v1/backups/' . $uuid . '/download');
    }

    /**
     * Make API request
     *
     * @param string $method HTTP method
     * @param string $endpoint API endpoint
     * @param array $params Request parameters
     * @return array|WP_Error
     */
    private function request($method, $endpoint, $params = array()) {
        if (empty($this->api_key)) {
            return new WP_Error('lox_auth_missing_key', __('API key not configured', 'lox-backup'));
        }

        $url = $this->base_url . $endpoint;

        if ($method === 'GET' && !empty($params)) {
            $url = add_query_arg($params, $url);
        }

        $args = array(
            'method' => $method,
            'timeout' => 30,
            'headers' => array(
                'X-API-Key' => $this->api_key,
                'Content-Type' => 'application/json',
                'User-Agent' => 'LOX-WordPress/' . LOX_BACKUP_VERSION,
            ),
        );

        if ($method !== 'GET' && !empty($params)) {
            $args['body'] = json_encode($params);
        }

        $response = wp_remote_request($url, $args);

        return $this->handle_response($response);
    }

    /**
     * Handle API response
     *
     * @param array|WP_Error $response
     * @return array|WP_Error
     */
    private function handle_response($response) {
        if (is_wp_error($response)) {
            // Map connection errors to standard codes
            $error_code = $response->get_error_code();
            if (strpos($error_code, 'http_request_failed') !== false) {
                return new WP_Error('lox_conn_failed', $response->get_error_message());
            }
            return $response;
        }

        $status_code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);

        if ($status_code === 401) {
            return new WP_Error('lox_auth_invalid_key', __('Authentication failed. Check your API key.', 'lox-backup'));
        }

        if ($status_code === 403) {
            return new WP_Error('lox_auth_forbidden', __('Access denied', 'lox-backup'));
        }

        if ($status_code === 404) {
            return new WP_Error('lox_backup_not_found', __('Resource not found', 'lox-backup'));
        }

        if ($status_code === 413) {
            $message = isset($data['detail']) ? $data['detail'] : '';
            if (strpos(strtolower($message), 'quota') !== false) {
                return new WP_Error('lox_upload_quota_exceeded', __('Storage quota exceeded', 'lox-backup'));
            }
            return new WP_Error('lox_upload_file_too_large', __('File too large', 'lox-backup'));
        }

        if ($status_code === 429) {
            return new WP_Error('lox_rate_exceeded', __('Rate limit exceeded. Please try again later.', 'lox-backup'));
        }

        if ($status_code >= 400) {
            $message = isset($data['detail']) ? $data['detail'] : __('API request failed', 'lox-backup');
            return new WP_Error('lox_upload_failed', $message);
        }

        return $data;
    }
}
