/**
 * LOX Backup Admin JavaScript
 * Modern UI interactions with smooth animations
 */

(function($) {
    'use strict';

    var LoxBackupAdmin = {
        init: function() {
            this.bindEvents();
            this.initAnimations();
        },

        bindEvents: function() {
            $('#lox-test-connection').on('click', this.testConnection);
            $('#lox-run-backup').on('click', this.runBackup);

            // Schedule form interactions
            $('.lox-frequency-select').on('change', this.handleFrequencyChange);

            // Toggle visibility of day selector based on frequency
            this.initScheduleForm();
        },

        initAnimations: function() {
            // Fade in cards on load
            $('.status-item, .lox-card, .lox-quick-stat').each(function(index) {
                $(this).css({
                    'opacity': '0',
                    'transform': 'translateY(20px)'
                }).delay(index * 100).animate({
                    'opacity': '1'
                }, 300, function() {
                    $(this).css('transform', 'translateY(0)');
                });
            });
        },

        initScheduleForm: function() {
            // Show/hide day selector based on frequency
            $('.lox-frequency-select').each(function() {
                var $row = $(this).closest('tr');
                var $daySelect = $row.find('.lox-day-select');

                if ($(this).val() === 'weekly') {
                    $daySelect.show();
                } else {
                    $daySelect.hide();
                }
            });
        },

        handleFrequencyChange: function() {
            var $row = $(this).closest('tr');
            var $daySelect = $row.find('.lox-day-select');

            if ($(this).val() === 'weekly') {
                $daySelect.fadeIn(200);
            } else {
                $daySelect.fadeOut(200);
            }
        },

        showMessage: function(message, type, duration) {
            var $container = $('#lox-backup-message');

            $container
                .removeClass('notice-success notice-error notice-warning notice-info')
                .addClass('notice-' + type)
                .html('<p>' + message + '</p>')
                .css('opacity', '0')
                .show()
                .animate({ opacity: 1 }, 200);

            // Auto-hide success messages after duration
            if (duration && type === 'success') {
                setTimeout(function() {
                    LoxBackupAdmin.hideMessage();
                }, duration);
            }
        },

        hideMessage: function() {
            $('#lox-backup-message').animate({ opacity: 0 }, 200, function() {
                $(this).hide();
            });
        },

        showModal: function(title, message, type) {
            // Remove existing modal
            $('#lox-modal-overlay').remove();

            var iconClass = type === 'success' ? 'dashicons-yes-alt' : 'dashicons-warning';
            var iconColor = type === 'success' ? '#10b981' : '#ef4444';
            var borderColor = type === 'success' ? '#10b981' : '#ef4444';

            var modalHtml =
                '<div id="lox-modal-overlay" class="lox-modal-overlay">' +
                    '<div class="lox-modal" style="border-top: 4px solid ' + borderColor + ';">' +
                        '<div class="lox-modal-header">' +
                            '<span class="dashicons ' + iconClass + '" style="color: ' + iconColor + '; font-size: 32px; width: 32px; height: 32px;"></span>' +
                            '<h2>' + title + '</h2>' +
                        '</div>' +
                        '<div class="lox-modal-body">' + message + '</div>' +
                        '<div class="lox-modal-footer">' +
                            '<button type="button" class="button button-primary lox-modal-close">OK</button>' +
                        '</div>' +
                    '</div>' +
                '</div>';

            $('body').append(modalHtml);

            // Animate in
            setTimeout(function() {
                $('#lox-modal-overlay').addClass('lox-modal-visible');
            }, 10);

            // Bind close events
            $('#lox-modal-overlay').on('click', function(e) {
                if (e.target === this) {
                    LoxBackupAdmin.closeModal();
                }
            });
            $('.lox-modal-close').on('click', function() {
                LoxBackupAdmin.closeModal();
            });
            $(document).on('keydown.loxmodal', function(e) {
                if (e.key === 'Escape') {
                    LoxBackupAdmin.closeModal();
                }
            });
        },

        closeModal: function() {
            $('#lox-modal-overlay').removeClass('lox-modal-visible');
            setTimeout(function() {
                $('#lox-modal-overlay').remove();
            }, 300);
            $(document).off('keydown.loxmodal');
        },

        setButtonLoading: function($button, loadingText) {
            var originalHtml = $button.html();
            $button.data('original-html', originalHtml);

            $button
                .prop('disabled', true)
                .addClass('lox-backup-running')
                .html('<span class="lox-spinner"></span> ' + loadingText);

            return originalHtml;
        },

        resetButton: function($button) {
            var originalHtml = $button.data('original-html');

            $button
                .prop('disabled', false)
                .removeClass('lox-backup-running')
                .html(originalHtml);
        },

        testConnection: function(e) {
            e.preventDefault();

            var $button = $(this);
            LoxBackupAdmin.setButtonLoading($button, loxBackup.strings.testing);
            LoxBackupAdmin.hideMessage();

            $.ajax({
                url: loxBackup.ajax_url,
                type: 'POST',
                data: {
                    action: 'lox_backup_test_connection',
                    nonce: loxBackup.nonce
                },
                success: function(response) {
                    if (response.success) {
                        LoxBackupAdmin.showModal(
                            loxBackup.strings.success || 'Connection Successful',
                            '<p>Successfully connected to LOX Backup Service.</p>' +
                            '<p><strong>Tenant:</strong> ' + response.data.tenant + '</p>',
                            'success'
                        );

                        // Add success animation to button
                        $button.addClass('lox-success-flash');
                        setTimeout(function() {
                            $button.removeClass('lox-success-flash');
                        }, 1000);
                    } else {
                        LoxBackupAdmin.showModal(
                            loxBackup.strings.error || 'Connection Failed',
                            '<p>' + response.data.message + '</p>',
                            'error'
                        );
                    }
                },
                error: function(xhr, status, error) {
                    var errorMsg = loxBackup.strings.error;
                    if (error) {
                        errorMsg += ': ' + error;
                    }
                    LoxBackupAdmin.showMessage(
                        '<span class="dashicons dashicons-warning"></span> ' + errorMsg,
                        'error'
                    );
                },
                complete: function() {
                    LoxBackupAdmin.resetButton($button);
                }
            });
        },

        runBackup: function(e) {
            e.preventDefault();

            // Custom confirmation dialog
            if (!confirm(loxBackup.strings.confirm_backup || 'Start backup now? This may take several minutes.')) {
                return;
            }

            var $button = $(this);
            LoxBackupAdmin.setButtonLoading($button, loxBackup.strings.backing_up);
            LoxBackupAdmin.hideMessage();

            // Show progress indicator
            LoxBackupAdmin.showMessage(
                '<span class="lox-spinner"></span> ' +
                'Backup in progress... Please don\'t close this page.',
                'info'
            );

            $.ajax({
                url: loxBackup.ajax_url,
                type: 'POST',
                timeout: 3600000, // 1 hour
                data: {
                    action: 'lox_backup_run_now',
                    nonce: loxBackup.nonce
                },
                success: function(response) {
                    if (response.success) {
                        LoxBackupAdmin.showMessage(
                            '<span class="dashicons dashicons-yes-alt" style="color: #10b981; font-size: 20px;"></span> ' +
                            '<strong>' + loxBackup.strings.backup_complete + '</strong>' +
                            '<br><small>UUID: <code>' + response.data.uuid + '</code></small>' +
                            '<br><small>Size: <strong>' + response.data.size + '</strong></small>',
                            'success'
                        );

                        // Reload page to update status
                        setTimeout(function() {
                            location.reload();
                        }, 3000);
                    } else {
                        LoxBackupAdmin.showMessage(
                            '<span class="dashicons dashicons-warning" style="color: #ef4444;"></span> ' +
                            loxBackup.strings.backup_failed + ': ' + response.data.message,
                            'error'
                        );
                    }
                },
                error: function(xhr, status, error) {
                    var message = loxBackup.strings.backup_failed;
                    if (status === 'timeout') {
                        message += ': Timeout - backup may still be running in background. Check the Backups page for status.';
                    } else if (error) {
                        message += ': ' + error;
                    }
                    LoxBackupAdmin.showMessage(
                        '<span class="dashicons dashicons-warning"></span> ' + message,
                        'error'
                    );
                },
                complete: function() {
                    LoxBackupAdmin.resetButton($button);
                }
            });
        },

        // Utility: Format file size
        formatBytes: function(bytes) {
            if (bytes === 0) return '0 B';
            var k = 1024;
            var sizes = ['B', 'KB', 'MB', 'GB', 'TB'];
            var i = Math.floor(Math.log(bytes) / Math.log(k));
            return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + ' ' + sizes[i];
        },

        // Utility: Format date
        formatDate: function(dateStr) {
            var date = new Date(dateStr);
            var options = {
                year: 'numeric',
                month: 'short',
                day: 'numeric',
                hour: '2-digit',
                minute: '2-digit'
            };
            return date.toLocaleDateString(undefined, options);
        },

        // Utility: Time ago
        timeAgo: function(dateStr) {
            var date = new Date(dateStr);
            var now = new Date();
            var seconds = Math.floor((now - date) / 1000);

            var intervals = {
                year: 31536000,
                month: 2592000,
                week: 604800,
                day: 86400,
                hour: 3600,
                minute: 60
            };

            for (var unit in intervals) {
                var interval = Math.floor(seconds / intervals[unit]);
                if (interval >= 1) {
                    return interval + ' ' + unit + (interval > 1 ? 's' : '') + ' ago';
                }
            }

            return 'Just now';
        }
    };

    // Additional styles injected for animations
    $('<style>')
        .text(`
            .lox-success-flash {
                animation: lox-success-flash 0.5s ease;
            }
            @keyframes lox-success-flash {
                0%, 100% { box-shadow: 0 0 0 0 rgba(16, 185, 129, 0); }
                50% { box-shadow: 0 0 0 8px rgba(16, 185, 129, 0.3); }
            }
            .status-item, .lox-card, .lox-quick-stat {
                transition: opacity 0.3s ease, transform 0.3s ease;
            }
            #lox-backup-message {
                transition: opacity 0.2s ease;
            }
            #lox-backup-message code {
                background: rgba(0,0,0,0.05);
                padding: 2px 6px;
                border-radius: 4px;
                font-size: 12px;
            }
            #lox-backup-message .dashicons {
                vertical-align: middle;
                margin-right: 4px;
            }
            /* Modal styles */
            .lox-modal-overlay {
                position: fixed;
                top: 0;
                left: 0;
                right: 0;
                bottom: 0;
                background: rgba(0, 0, 0, 0.5);
                display: flex;
                align-items: center;
                justify-content: center;
                z-index: 100000;
                opacity: 0;
                transition: opacity 0.3s ease;
            }
            .lox-modal-overlay.lox-modal-visible {
                opacity: 1;
            }
            .lox-modal {
                background: #fff;
                border-radius: 8px;
                box-shadow: 0 20px 50px rgba(0, 0, 0, 0.3);
                max-width: 420px;
                width: 90%;
                transform: scale(0.9) translateY(-20px);
                transition: transform 0.3s ease;
                overflow: hidden;
            }
            .lox-modal-visible .lox-modal {
                transform: scale(1) translateY(0);
            }
            .lox-modal-header {
                padding: 24px 24px 16px;
                text-align: center;
            }
            .lox-modal-header h2 {
                margin: 12px 0 0;
                font-size: 20px;
                font-weight: 600;
                color: #1e293b;
            }
            .lox-modal-body {
                padding: 0 24px 24px;
                text-align: center;
                color: #64748b;
                font-size: 14px;
                line-height: 1.6;
            }
            .lox-modal-body p {
                margin: 8px 0;
            }
            .lox-modal-footer {
                padding: 16px 24px;
                background: #f8fafc;
                text-align: center;
                border-top: 1px solid #e2e8f0;
            }
            .lox-modal-footer .button {
                min-width: 100px;
            }
            /* Tabs styles */
            .lox-tabs {
                display: flex;
                gap: 4px;
                margin: 16px 0;
                border-bottom: 2px solid #e2e8f0;
                padding-bottom: 0;
            }
            .lox-tab {
                padding: 12px 20px;
                border: none;
                background: transparent;
                cursor: pointer;
                font-size: 14px;
                font-weight: 500;
                color: #64748b;
                border-bottom: 2px solid transparent;
                margin-bottom: -2px;
                display: flex;
                align-items: center;
                gap: 6px;
                transition: all 0.2s ease;
            }
            .lox-tab:hover {
                color: #3b82f6;
            }
            .lox-tab.active {
                color: #3b82f6;
                border-bottom-color: #3b82f6;
            }
            .lox-tab .dashicons {
                font-size: 16px;
                width: 16px;
                height: 16px;
            }
            .lox-tab-content {
                display: none !important;
            }
            .lox-tab-content.active:not(.lox-hidden) {
                display: block !important;
            }
            /* Profiles grid */
            .lox-profiles-grid {
                display: grid;
                grid-template-columns: repeat(auto-fill, minmax(340px, 1fr));
                gap: 16px;
                margin-top: 16px;
            }
            .lox-profile-card {
                background: #fff;
                border: 1px solid #e2e8f0;
                border-radius: 8px;
                padding: 20px;
                border-left: 4px solid #3b82f6;
                transition: box-shadow 0.2s ease;
            }
            .lox-profile-card:hover {
                box-shadow: 0 4px 12px rgba(0, 0, 0, 0.1);
            }
            .lox-profile-header {
                display: flex;
                align-items: flex-start;
                gap: 12px;
                margin-bottom: 16px;
            }
            .lox-profile-icon {
                width: 40px;
                height: 40px;
                background: #eff6ff;
                border-radius: 8px;
                display: flex;
                align-items: center;
                justify-content: center;
            }
            .lox-profile-icon .dashicons {
                color: #3b82f6;
                font-size: 20px;
                width: 20px;
                height: 20px;
            }
            .lox-profile-info h3 {
                margin: 0 0 4px;
                font-size: 15px;
                font-weight: 600;
                color: #1e293b;
            }
            .lox-profile-badges {
                display: flex;
                gap: 6px;
            }
            .lox-profile-badge {
                font-size: 11px;
                padding: 2px 8px;
                background: #f1f5f9;
                border-radius: 4px;
                color: #64748b;
                text-transform: capitalize;
            }
            .lox-profile-last-backup {
                background: #f8fafc;
                border-radius: 6px;
                padding: 12px;
                margin-bottom: 16px;
            }
            .lox-profile-last-backup-header {
                display: flex;
                justify-content: space-between;
                align-items: center;
            }
            .lox-profile-last-backup-info {
                display: flex;
                align-items: center;
                gap: 8px;
                font-size: 13px;
                color: #475569;
            }
            .lox-profile-last-backup-info .dashicons {
                font-size: 16px;
                width: 16px;
                height: 16px;
            }
            .lox-profile-last-backup-info .dashicons-yes-alt { color: #10b981; }
            .lox-profile-last-backup-info .dashicons-dismiss { color: #ef4444; }
            .lox-profile-last-backup-info .dashicons-clock { color: #f59e0b; }
            .lox-profile-status {
                font-size: 11px;
                font-weight: 500;
                text-transform: capitalize;
            }
            .lox-profile-status.completed { color: #10b981; }
            .lox-profile-status.failed { color: #ef4444; }
            .lox-profile-status.distributing,
            .lox-profile-status.validating { color: #f59e0b; }
            .lox-profile-stats {
                display: grid;
                grid-template-columns: repeat(3, 1fr);
                gap: 12px;
                padding-top: 16px;
                border-top: 1px solid #e2e8f0;
            }
            .lox-profile-stat {
                text-align: center;
            }
            .lox-profile-stat-label {
                font-size: 11px;
                color: #94a3b8;
                text-transform: uppercase;
                letter-spacing: 0.5px;
            }
            .lox-profile-stat-value {
                font-size: 16px;
                font-weight: 600;
                color: #1e293b;
                margin-top: 2px;
            }
            .lox-profile-failures {
                margin-top: 12px;
                padding: 8px 12px;
                background: #fef2f2;
                border-radius: 6px;
                display: flex;
                align-items: center;
                gap: 8px;
                font-size: 13px;
                color: #dc2626;
            }
            .lox-profile-failures .dashicons {
                font-size: 16px;
                width: 16px;
                height: 16px;
            }
        `)
        .appendTo('head');

    // Tab switching and profiles loading
    var LoxBackupProfiles = {
        init: function() {
            console.log('LOX: LoxBackupProfiles.init() called');
            this.bindTabEvents();
            // Load profiles on page load if on backups page
            if ($('#lox-profiles-tab').length) {
                console.log('LOX: #lox-profiles-tab found, loading profiles...');
                this.loadProfiles();
            } else {
                console.log('LOX: #lox-profiles-tab NOT found');
            }
        },

        bindTabEvents: function() {
            var self = this;

            $('.lox-tab').on('click', function() {
                var tab = $(this).data('tab');

                // Update active tab
                $('.lox-tab').removeClass('active');
                $(this).addClass('active');

                // Show corresponding content using class-based approach
                $('.lox-tab-content').removeClass('active').addClass('lox-hidden');
                $('#lox-' + tab + '-tab').addClass('active').removeClass('lox-hidden');

                // Load data for the tab
                if (tab === 'profiles') {
                    self.loadProfiles();
                } else if (tab === 'activity') {
                    self.loadBackups();
                }
            });

            // Click on profile card to view its backups
            $(document).on('click', '.lox-profile-card', function() {
                var profileId = $(this).data('profile-id');
                var profileName = $(this).data('profile-name');

                if (profileId) {
                    // Store profile name for display
                    self.currentProfileName = profileName;

                    // Switch to Activity tab
                    $('.lox-tab').removeClass('active');
                    $('.lox-tab[data-tab="activity"]').addClass('active');
                    $('.lox-tab-content').removeClass('active').addClass('lox-hidden');
                    $('#lox-activity-tab').addClass('active').removeClass('lox-hidden');

                    // Load backups filtered by profile
                    self.loadBackups(profileId);
                }
            });

            // Clear filter link
            $(document).on('click', '.lox-clear-filter', function(e) {
                e.preventDefault();
                self.loadBackups(); // Load without filter
            });
        },

        loadProfiles: function() {
            var self = this;
            var $loading = $('#lox-profiles-loading');
            var $grid = $('#lox-profiles-grid');
            var $empty = $('#lox-profiles-empty');

            // Use class-based show/hide for reliable override of CSS display: flex
            $loading.removeClass('lox-hidden');
            $grid.addClass('lox-hidden');
            $empty.addClass('lox-hidden');

            $.ajax({
                url: loxBackup.ajax_url,
                type: 'POST',
                data: {
                    action: 'lox_backup_get_remote_profiles',
                    nonce: loxBackup.nonce
                },
                success: function(response) {
                    $loading.addClass('lox-hidden');

                    // Defensive check for response structure
                    var data = response.data || {};
                    var profiles = data.profiles || [];

                    if (response.success && profiles.length > 0) {
                        self.renderProfiles(profiles);
                        $grid.removeClass('lox-hidden');
                    } else {
                        $empty.removeClass('lox-hidden');
                    }
                },
                error: function(xhr, status, error) {
                    console.error('LOX Profiles load error:', status, error);
                    $loading.addClass('lox-hidden');
                    $empty.removeClass('lox-hidden');
                }
            });
        },

        currentProfileFilter: null,

        loadBackups: function(profileId) {
            var self = this;
            var $loading = $('#lox-backups-loading');
            var $table = $('#lox-backups-table');
            var $empty = $('#lox-backups-empty');
            var $filter = $('#lox-backups-filter');

            // Store current filter
            self.currentProfileFilter = profileId || null;

            // Use class-based show/hide for reliable override of CSS display: flex
            $loading.removeClass('lox-hidden');
            $table.addClass('lox-hidden');
            $empty.addClass('lox-hidden');

            var requestData = {
                action: 'lox_backup_list',
                nonce: loxBackup.nonce,
                page: 1
            };

            // Add profile filter if specified
            if (profileId) {
                requestData.profile_id = profileId;
            }

            $.ajax({
                url: loxBackup.ajax_url,
                type: 'POST',
                data: requestData,
                success: function(response) {
                    $loading.addClass('lox-hidden');

                    // API returns 'items' not 'backups' - defensive check for data structure
                    var data = response.data || {};
                    var backups = data.items || data.backups || [];
                    if (response.success && backups.length > 0) {
                        var $tbody = $('#lox-backups-list').empty();

                        backups.forEach(function(backup) {
                            var statusClass = backup.status === 'completed' ? 'success' :
                                             backup.status === 'failed' ? 'error' : 'warning';
                            var versionText = backup.version ? 'v' + backup.version : '-';

                            $tbody.append(
                                '<tr data-profile-id="' + (backup.profile_id || '') + '">' +
                                '<td><strong>' + self.escapeHtml(backup.name) + '</strong><br><small class="lox-uuid">' + backup.uuid + '</small></td>' +
                                '<td><span class="lox-version-badge">' + versionText + '</span></td>' +
                                '<td><span class="lox-component-badge">' + (backup.component || 'full') + '</span></td>' +
                                '<td>' + LoxBackupAdmin.formatBytes(backup.size_bytes) + '</td>' +
                                '<td><span class="lox-status ' + statusClass + '">' + backup.status + '</span></td>' +
                                '<td>' + LoxBackupAdmin.timeAgo(backup.created_at) + '</td>' +
                                '<td class="lox-actions">' +
                                    '<button class="button button-small lox-restore-btn" data-uuid="' + backup.uuid + '" data-name="' + backup.name + '">' +
                                    '<span class="dashicons dashicons-backup"></span> Restore</button>' +
                                '</td>' +
                                '</tr>'
                            );
                        });

                        $table.removeClass('lox-hidden');

                        // Show/hide filter indicator
                        if (profileId) {
                            var filterText = self.currentProfileName ?
                                'Filtered by: <strong>' + self.escapeHtml(self.currentProfileName) + '</strong>' :
                                'Filtered by profile';
                            $('#lox-backups-filter').remove();
                            $('#lox-backups-table').before(
                                '<div id="lox-backups-filter" class="lox-filter-notice">' +
                                '<span class="dashicons dashicons-filter"></span> ' +
                                filterText + ' • <a href="#" class="lox-clear-filter">Show all</a>' +
                                '</div>'
                            );
                        } else {
                            $('#lox-backups-filter').remove();
                        }
                    } else {
                        $empty.removeClass('lox-hidden');
                        $('#lox-backups-filter').remove();
                    }
                },
                error: function() {
                    $loading.addClass('lox-hidden');
                    $empty.removeClass('lox-hidden');
                }
            });
        },

        renderProfiles: function(profiles) {
            var $grid = $('#lox-profiles-grid').empty();
            var self = this;

            profiles.forEach(function(profile) {
                var statusIcon = profile.last_backup_status === 'completed' ? 'dashicons-yes-alt' :
                                profile.last_backup_status === 'failed' ? 'dashicons-dismiss' : 'dashicons-clock';
                var statusClass = (profile.last_backup_status || '').toLowerCase();

                var lastBackupHtml = '';
                if (profile.last_backup_at) {
                    lastBackupHtml =
                        '<div class="lox-profile-last-backup">' +
                            '<div class="lox-profile-last-backup-header">' +
                                '<div class="lox-profile-last-backup-info">' +
                                    '<span class="dashicons ' + statusIcon + '"></span>' +
                                    '<span>v' + profile.current_version + '</span>' +
                                    '<span>&bull;</span>' +
                                    '<span>' + LoxBackupAdmin.formatBytes(profile.last_backup_size_bytes || 0) + '</span>' +
                                    '<span>&bull;</span>' +
                                    '<span>' + LoxBackupAdmin.timeAgo(profile.last_backup_at) + '</span>' +
                                '</div>' +
                                '<span class="lox-profile-status ' + statusClass + '">' + statusClass + '</span>' +
                            '</div>' +
                        '</div>';
                }

                var failuresHtml = '';
                if (profile.consecutive_failures > 0) {
                    failuresHtml =
                        '<div class="lox-profile-failures">' +
                            '<span class="dashicons dashicons-warning"></span>' +
                            profile.consecutive_failures + ' consecutive failure' + (profile.consecutive_failures > 1 ? 's' : '') +
                        '</div>';
                }

                var componentIcon = profile.component === 'database' ? 'dashicons-database' :
                                   profile.component === 'files' ? 'dashicons-portfolio' :
                                   profile.component === 'uploads' ? 'dashicons-format-image' : 'dashicons-media-archive';

                $grid.append(
                    '<div class="lox-profile-card" data-profile-id="' + profile.uuid + '" data-profile-name="' + self.escapeHtml(profile.name) + '">' +
                        '<div class="lox-profile-header">' +
                            '<div class="lox-profile-icon">' +
                                '<span class="dashicons ' + componentIcon + '"></span>' +
                            '</div>' +
                            '<div class="lox-profile-info">' +
                                '<h3>' + self.escapeHtml(profile.name) + '</h3>' +
                                '<div class="lox-profile-badges">' +
                                    (profile.source ? '<span class="lox-profile-badge">' + profile.source + '</span>' : '') +
                                    (profile.component ? '<span class="lox-profile-badge">' + profile.component + '</span>' : '') +
                                '</div>' +
                            '</div>' +
                        '</div>' +
                        lastBackupHtml +
                        '<div class="lox-profile-stats">' +
                            '<div class="lox-profile-stat">' +
                                '<div class="lox-profile-stat-label">Versions</div>' +
                                '<div class="lox-profile-stat-value">' + profile.current_version + '</div>' +
                            '</div>' +
                            '<div class="lox-profile-stat">' +
                                '<div class="lox-profile-stat-label">Total Size</div>' +
                                '<div class="lox-profile-stat-value">' + LoxBackupAdmin.formatBytes(profile.total_size_bytes) + '</div>' +
                            '</div>' +
                            '<div class="lox-profile-stat">' +
                                '<div class="lox-profile-stat-label">Backups</div>' +
                                '<div class="lox-profile-stat-value">' + profile.backup_count + '</div>' +
                            '</div>' +
                        '</div>' +
                        failuresHtml +
                        '<div class="lox-profile-action">' +
                            '<span class="dashicons dashicons-arrow-right-alt2"></span> View backups' +
                        '</div>' +
                    '</div>'
                );
            });
        },

        escapeHtml: function(text) {
            var div = document.createElement('div');
            div.textContent = text;
            return div.innerHTML;
        }
    };

    $(document).ready(function() {
        console.log('LOX: admin.js document ready - v2.0');
        LoxBackupAdmin.init();
        LoxBackupProfiles.init();
    });

    // Expose for external use
    window.LoxBackupAdmin = LoxBackupAdmin;

})(jQuery);
