<?php
/**
 * LOX WooCommerce Backup Handler
 *
 * Extends the base LOX_Backup class with WooCommerce-specific components.
 *
 * @package LOX_Backup_WooCommerce
 */

if (!defined('ABSPATH')) {
    exit;
}

/**
 * WooCommerce-specific backup handler
 */
class LOX_WC_Backup_Handler extends LOX_Backup {

    /**
     * Source identifier
     */
    protected $source = 'woocommerce';

    /**
     * Component definitions - extends WordPress with WooCommerce specifics
     */
    protected $components = array(
        'content' => array(
            'name' => 'Content',
            'description' => 'Posts, pages, and media metadata',
            'tables' => array('posts', 'postmeta', 'terms', 'term_taxonomy', 'term_relationships', 'termmeta'),
        ),
        'catalog' => array(
            'name' => 'Catalog',
            'description' => 'Products, categories, attributes, and variations',
            'tables' => array(
                'wc_product_meta_lookup',
                'wc_product_attributes_lookup',
                'wc_reserved_stock',
            ),
            'post_types' => array('product', 'product_variation'),
            'taxonomies' => array('product_cat', 'product_tag', 'product_type', 'product_visibility', 'product_shipping_class'),
        ),
        'transactional' => array(
            'name' => 'Orders & Customers',
            'description' => 'Orders, customers, and payment data',
            'tables' => array(
                'wc_orders',
                'wc_orders_meta',
                'wc_order_items',
                'wc_order_itemmeta',
                'wc_order_stats',
                'wc_customer_lookup',
                'wc_order_addresses',
                'wc_order_operational_data',
                'users',
                'usermeta',
            ),
            'post_types' => array('shop_order', 'shop_order_refund', 'shop_subscription'),
        ),
        'files' => array(
            'name' => 'Media Files',
            'description' => 'Product images and downloads',
            'paths' => array('uploads'),
        ),
        'config' => array(
            'name' => 'Configuration',
            'description' => 'Store settings, shipping, taxes, and payment gateways',
            'tables' => array(
                'options',
                'woocommerce_tax_rates',
                'woocommerce_tax_rate_locations',
                'woocommerce_shipping_zones',
                'woocommerce_shipping_zone_locations',
                'woocommerce_shipping_zone_methods',
                'woocommerce_payment_tokens',
                'woocommerce_payment_tokenmeta',
            ),
            'paths' => array('plugins', 'themes'),
            'files' => array('wp-config.php'),
        ),
    );

    /**
     * Run WooCommerce component backup
     *
     * @param string $component Component name
     * @return array|WP_Error Result
     */
    public function run_wc_component_backup($component) {
        if (!isset($this->components[$component])) {
            return new WP_Error('invalid_component', __('Invalid backup component', 'lox-backup-woocommerce'));
        }

        $start_time = time();
        $timestamp = date('Ymd_His');
        $site_name = sanitize_title(get_bloginfo('name'));
        $backup_name = "{$this->source}-{$site_name}-{$component}-{$timestamp}";
        $component_def = $this->components[$component];

        $this->log("Starting WooCommerce {$component} backup: {$backup_name}");

        try {
            // Create temp directory
            $temp_dir = $this->get_backup_dir_path() . '/temp_wc_' . $component . '_' . $timestamp;
            wp_mkdir_p($temp_dir);

            // Backup tables if defined
            if (!empty($component_def['tables'])) {
                $db_file = $this->backup_wc_tables($temp_dir, $component_def['tables']);
                if (is_wp_error($db_file)) {
                    $this->cleanup($temp_dir);
                    return $db_file;
                }
                $this->log("Database tables backed up for {$component}");
            }

            // Backup post types if defined
            if (!empty($component_def['post_types'])) {
                $posts_file = $this->backup_post_types($temp_dir, $component_def['post_types']);
                if (!is_wp_error($posts_file)) {
                    $this->log("Post types backed up for {$component}");
                }
            }

            // Backup taxonomies if defined
            if (!empty($component_def['taxonomies'])) {
                $terms_file = $this->backup_taxonomies($temp_dir, $component_def['taxonomies']);
                if (!is_wp_error($terms_file)) {
                    $this->log("Taxonomies backed up for {$component}");
                }
            }

            // Export WooCommerce data as JSON if applicable
            if ($component === 'catalog') {
                $this->export_catalog_data($temp_dir);
            } elseif ($component === 'transactional') {
                $this->export_transactional_data($temp_dir);
            } elseif ($component === 'config') {
                $this->export_config_data($temp_dir);
            }

            // Backup paths if defined
            if (!empty($component_def['paths'])) {
                foreach ($component_def['paths'] as $path) {
                    $full_path = WP_CONTENT_DIR . '/' . $path;
                    if (is_dir($full_path)) {
                        $this->backup_directory($full_path, $temp_dir, $path);
                        $this->log("Directory {$path} backed up");
                    }
                }
            }

            // Backup specific files
            if (!empty($component_def['files'])) {
                foreach ($component_def['files'] as $file) {
                    if ($file === 'wp-config.php') {
                        $this->backup_wp_config($temp_dir);
                    }
                }
            }

            // Create archive
            $archive_path = $this->get_backup_dir_path() . '/' . $backup_name . '.tar.gz';
            $this->create_archive($temp_dir, $archive_path);
            $this->log("Archive created: " . $this->format_bytes(filesize($archive_path)));

            // Get WooCommerce site metadata
            $extra_metadata = $this->get_wc_site_metadata();
            $extra_metadata['backup_component'] = $component;

            // Upload to LOX with metadata
            $api = new LOX_API();
            $result = $api->upload_backup($archive_path, array(
                'name' => $backup_name,
                'tags' => "{$this->source},{$component},automated",
                'description' => sprintf(
                    'WooCommerce %s backup from %s (%s)',
                    $component_def['name'],
                    home_url(),
                    get_bloginfo('name')
                ),
                'source' => $this->source,
                'component' => $component,
                'extra_metadata' => $extra_metadata,
            ));

            if (is_wp_error($result)) {
                $this->cleanup($temp_dir, $archive_path);
                $this->update_component_status($component, 'failed', $result->get_error_message());
                return $result;
            }

            // Wait for completion
            $backup = $api->wait_for_completion($result['uuid'], 1800);

            // Cleanup
            $this->cleanup($temp_dir, $archive_path);

            if (is_wp_error($backup)) {
                $this->update_component_status($component, 'failed', $backup->get_error_message());
                return $backup;
            }

            $duration = time() - $start_time;
            $this->log(sprintf('WooCommerce %s backup completed in %d seconds. UUID: %s', $component, $duration, $backup['uuid']));
            $this->update_component_status($component, 'completed', null, $backup);

            return $backup;

        } catch (Exception $e) {
            $this->log("WooCommerce backup failed: " . $e->getMessage(), 'error');
            $this->update_component_status($component, 'failed', $e->getMessage());
            return new WP_Error('backup_failed', $e->getMessage());
        }
    }

    /**
     * Backup WooCommerce-specific tables
     *
     * @param string $temp_dir Temp directory
     * @param array $tables Table names (with or without prefix)
     * @return string|WP_Error SQL file path
     */
    private function backup_wc_tables($temp_dir, $tables) {
        global $wpdb;

        $sql_file = $temp_dir . '/database.sql';
        $handle = fopen($sql_file, 'w');

        if (!$handle) {
            return new WP_Error('db_backup_failed', __('Failed to create database backup file', 'lox-backup-woocommerce'));
        }

        fwrite($handle, "-- LOX WooCommerce Component Backup\n");
        fwrite($handle, "-- Generated: " . date('Y-m-d H:i:s') . "\n");
        fwrite($handle, "-- Site: " . home_url() . "\n\n");
        fwrite($handle, "SET NAMES utf8mb4;\n");
        fwrite($handle, "SET FOREIGN_KEY_CHECKS = 0;\n\n");

        foreach ($tables as $table_suffix) {
            // Handle tables that may or may not have prefix
            $table_name = $wpdb->prefix . $table_suffix;

            // Check if table exists
            $exists = $wpdb->get_var($wpdb->prepare(
                "SHOW TABLES LIKE %s",
                $table_name
            ));

            if (!$exists) {
                continue;
            }

            // Table structure
            $create = $wpdb->get_row("SHOW CREATE TABLE `{$table_name}`", ARRAY_N);
            fwrite($handle, "DROP TABLE IF EXISTS `{$table_name}`;\n");
            fwrite($handle, $create[1] . ";\n\n");

            // Table data in chunks
            $offset = 0;
            $limit = 1000;

            while (true) {
                $rows = $wpdb->get_results(
                    $wpdb->prepare("SELECT * FROM `{$table_name}` LIMIT %d OFFSET %d", $limit, $offset),
                    ARRAY_A
                );

                if (empty($rows)) {
                    break;
                }

                $columns = array_keys($rows[0]);
                $column_list = '`' . implode('`, `', $columns) . '`';

                foreach (array_chunk($rows, 100) as $chunk) {
                    $values = array();
                    foreach ($chunk as $row) {
                        $escaped = array_map(function($v) use ($wpdb) {
                            return $v === null ? 'NULL' : "'" . $wpdb->_real_escape($v) . "'";
                        }, array_values($row));
                        $values[] = '(' . implode(', ', $escaped) . ')';
                    }
                    fwrite($handle, "INSERT INTO `{$table_name}` ({$column_list}) VALUES\n" . implode(",\n", $values) . ";\n");
                }

                $offset += $limit;
            }

            fwrite($handle, "\n");
        }

        fwrite($handle, "SET FOREIGN_KEY_CHECKS = 1;\n");
        fclose($handle);

        return $sql_file;
    }

    /**
     * Backup specific post types
     *
     * @param string $temp_dir Temp directory
     * @param array $post_types Post type names
     * @return string|WP_Error SQL file path
     */
    private function backup_post_types($temp_dir, $post_types) {
        global $wpdb;

        $sql_file = $temp_dir . '/posts.sql';
        $handle = fopen($sql_file, 'w');

        if (!$handle) {
            return new WP_Error('db_backup_failed', __('Failed to create posts backup file', 'lox-backup-woocommerce'));
        }

        fwrite($handle, "-- LOX WooCommerce Posts Backup\n");
        fwrite($handle, "-- Post types: " . implode(', ', $post_types) . "\n\n");
        fwrite($handle, "SET NAMES utf8mb4;\n\n");

        $post_types_escaped = array_map(function($pt) use ($wpdb) {
            return $wpdb->prepare('%s', $pt);
        }, $post_types);

        $posts_query = "SELECT * FROM {$wpdb->posts} WHERE post_type IN (" . implode(',', $post_types_escaped) . ")";
        $posts = $wpdb->get_results($posts_query, ARRAY_A);

        if (!empty($posts)) {
            $columns = array_keys($posts[0]);
            $column_list = '`' . implode('`, `', $columns) . '`';

            foreach (array_chunk($posts, 100) as $chunk) {
                $values = array();
                foreach ($chunk as $row) {
                    $escaped = array_map(function($v) use ($wpdb) {
                        return $v === null ? 'NULL' : "'" . $wpdb->_real_escape($v) . "'";
                    }, array_values($row));
                    $values[] = '(' . implode(', ', $escaped) . ')';
                }
                fwrite($handle, "INSERT INTO `{$wpdb->posts}` ({$column_list}) VALUES\n" . implode(",\n", $values) . ";\n");
            }

            // Also backup postmeta for these posts
            $post_ids = wp_list_pluck($posts, 'ID');
            if (!empty($post_ids)) {
                $meta_query = "SELECT * FROM {$wpdb->postmeta} WHERE post_id IN (" . implode(',', array_map('intval', $post_ids)) . ")";
                $metas = $wpdb->get_results($meta_query, ARRAY_A);

                if (!empty($metas)) {
                    fwrite($handle, "\n-- Post meta\n");
                    $columns = array_keys($metas[0]);
                    $column_list = '`' . implode('`, `', $columns) . '`';

                    foreach (array_chunk($metas, 100) as $chunk) {
                        $values = array();
                        foreach ($chunk as $row) {
                            $escaped = array_map(function($v) use ($wpdb) {
                                return $v === null ? 'NULL' : "'" . $wpdb->_real_escape($v) . "'";
                            }, array_values($row));
                            $values[] = '(' . implode(', ', $escaped) . ')';
                        }
                        fwrite($handle, "INSERT INTO `{$wpdb->postmeta}` ({$column_list}) VALUES\n" . implode(",\n", $values) . ";\n");
                    }
                }
            }
        }

        fclose($handle);
        return $sql_file;
    }

    /**
     * Backup specific taxonomies
     *
     * @param string $temp_dir Temp directory
     * @param array $taxonomies Taxonomy names
     * @return string|WP_Error SQL file path
     */
    private function backup_taxonomies($temp_dir, $taxonomies) {
        global $wpdb;

        $sql_file = $temp_dir . '/taxonomies.sql';
        $handle = fopen($sql_file, 'w');

        if (!$handle) {
            return new WP_Error('db_backup_failed', __('Failed to create taxonomies backup file', 'lox-backup-woocommerce'));
        }

        fwrite($handle, "-- LOX WooCommerce Taxonomies Backup\n");
        fwrite($handle, "-- Taxonomies: " . implode(', ', $taxonomies) . "\n\n");
        fwrite($handle, "SET NAMES utf8mb4;\n\n");

        $taxonomies_escaped = array_map(function($t) use ($wpdb) {
            return $wpdb->prepare('%s', $t);
        }, $taxonomies);

        // Get term_taxonomy entries
        $tt_query = "SELECT * FROM {$wpdb->term_taxonomy} WHERE taxonomy IN (" . implode(',', $taxonomies_escaped) . ")";
        $term_taxonomies = $wpdb->get_results($tt_query, ARRAY_A);

        if (!empty($term_taxonomies)) {
            $term_ids = wp_list_pluck($term_taxonomies, 'term_id');

            // Get terms
            $terms_query = "SELECT * FROM {$wpdb->terms} WHERE term_id IN (" . implode(',', array_map('intval', $term_ids)) . ")";
            $terms = $wpdb->get_results($terms_query, ARRAY_A);

            if (!empty($terms)) {
                $columns = array_keys($terms[0]);
                $column_list = '`' . implode('`, `', $columns) . '`';

                foreach (array_chunk($terms, 100) as $chunk) {
                    $values = array();
                    foreach ($chunk as $row) {
                        $escaped = array_map(function($v) use ($wpdb) {
                            return $v === null ? 'NULL' : "'" . $wpdb->_real_escape($v) . "'";
                        }, array_values($row));
                        $values[] = '(' . implode(', ', $escaped) . ')';
                    }
                    fwrite($handle, "INSERT INTO `{$wpdb->terms}` ({$column_list}) VALUES\n" . implode(",\n", $values) . ";\n");
                }
            }

            // Write term_taxonomy
            fwrite($handle, "\n-- Term taxonomy\n");
            $columns = array_keys($term_taxonomies[0]);
            $column_list = '`' . implode('`, `', $columns) . '`';

            foreach (array_chunk($term_taxonomies, 100) as $chunk) {
                $values = array();
                foreach ($chunk as $row) {
                    $escaped = array_map(function($v) use ($wpdb) {
                        return $v === null ? 'NULL' : "'" . $wpdb->_real_escape($v) . "'";
                    }, array_values($row));
                    $values[] = '(' . implode(', ', $escaped) . ')';
                }
                fwrite($handle, "INSERT INTO `{$wpdb->term_taxonomy}` ({$column_list}) VALUES\n" . implode(",\n", $values) . ";\n");
            }
        }

        fclose($handle);
        return $sql_file;
    }

    /**
     * Export catalog data as JSON
     *
     * @param string $temp_dir Temp directory
     */
    private function export_catalog_data($temp_dir) {
        $wc_dir = $temp_dir . '/woocommerce';
        wp_mkdir_p($wc_dir);

        LOX_WC_Export::export_products($wc_dir . '/products.json');
        $this->log('Exported products JSON');
    }

    /**
     * Export transactional data as JSON
     *
     * @param string $temp_dir Temp directory
     */
    private function export_transactional_data($temp_dir) {
        $wc_dir = $temp_dir . '/woocommerce';
        wp_mkdir_p($wc_dir);

        LOX_WC_Export::export_orders($wc_dir . '/orders.json');
        LOX_WC_Export::export_customers($wc_dir . '/customers.json');
        $this->log('Exported orders and customers JSON');
    }

    /**
     * Export config data as JSON
     *
     * @param string $temp_dir Temp directory
     */
    private function export_config_data($temp_dir) {
        $wc_dir = $temp_dir . '/woocommerce';
        wp_mkdir_p($wc_dir);

        LOX_WC_Export::export_settings($wc_dir . '/settings.json');
        $this->export_tax_rates($wc_dir . '/tax-rates.json');
        $this->export_shipping_zones($wc_dir . '/shipping-zones.json');
        $this->log('Exported settings, tax rates, and shipping zones');
    }

    /**
     * Export tax rates
     *
     * @param string $file_path
     */
    private function export_tax_rates($file_path) {
        global $wpdb;

        $rates = $wpdb->get_results("
            SELECT tr.*, GROUP_CONCAT(trl.location_code) as postcodes
            FROM {$wpdb->prefix}woocommerce_tax_rates tr
            LEFT JOIN {$wpdb->prefix}woocommerce_tax_rate_locations trl
                ON tr.tax_rate_id = trl.tax_rate_id AND trl.location_type = 'postcode'
            GROUP BY tr.tax_rate_id
        ", ARRAY_A);

        if (!empty($rates)) {
            file_put_contents($file_path, json_encode($rates, JSON_PRETTY_PRINT));
        }
    }

    /**
     * Export shipping zones
     *
     * @param string $file_path
     */
    private function export_shipping_zones($file_path) {
        if (!class_exists('WC_Shipping_Zones')) {
            return;
        }

        $zones_data = array();
        $zones = WC_Shipping_Zones::get_zones();

        foreach ($zones as $zone) {
            $zone_obj = new WC_Shipping_Zone($zone['zone_id']);
            $zone_data = array(
                'id' => $zone['zone_id'],
                'name' => $zone['zone_name'],
                'order' => $zone['zone_order'],
                'locations' => array(),
                'methods' => array(),
            );

            foreach ($zone_obj->get_zone_locations() as $location) {
                $zone_data['locations'][] = array(
                    'code' => $location->code,
                    'type' => $location->type,
                );
            }

            foreach ($zone_obj->get_shipping_methods() as $method) {
                $zone_data['methods'][] = array(
                    'instance_id' => $method->instance_id,
                    'method_id' => $method->id,
                    'title' => $method->title,
                    'enabled' => $method->enabled,
                    'settings' => $method->instance_settings,
                );
            }

            $zones_data[] = $zone_data;
        }

        if (!empty($zones_data)) {
            file_put_contents($file_path, json_encode($zones_data, JSON_PRETTY_PRINT));
        }
    }

    /**
     * Get backup directory path
     *
     * @return string
     */
    private function get_backup_dir_path() {
        $upload_dir = wp_upload_dir();
        $backup_dir = $upload_dir['basedir'] . '/lox-backups';

        if (!file_exists($backup_dir)) {
            wp_mkdir_p($backup_dir);
            file_put_contents($backup_dir . '/.htaccess', 'deny from all');
            file_put_contents($backup_dir . '/index.php', '<?php // Silence is golden');
        }

        return $backup_dir;
    }

    /**
     * Backup wp-config.php
     *
     * @param string $temp_dir Temp directory
     * @return string|WP_Error File path
     */
    private function backup_wp_config($temp_dir) {
        $config_path = ABSPATH . 'wp-config.php';

        if (!file_exists($config_path)) {
            $config_path = dirname(ABSPATH) . '/wp-config.php';
        }

        if (!file_exists($config_path)) {
            return new WP_Error('config_not_found', __('wp-config.php not found', 'lox-backup-woocommerce'));
        }

        $dest_path = $temp_dir . '/wp-config.php';
        copy($config_path, $dest_path);

        return $dest_path;
    }

    /**
     * Cleanup temporary files
     *
     * @param string $temp_dir Temp directory
     * @param string $archive Archive file
     */
    private function cleanup($temp_dir, $archive = null) {
        if (is_dir($temp_dir)) {
            $this->recursive_delete($temp_dir);
        }

        if ($archive && file_exists($archive)) {
            unlink($archive);
        }
    }

    /**
     * Recursively delete directory
     *
     * @param string $dir Directory path
     */
    private function recursive_delete($dir) {
        if (is_dir($dir)) {
            $objects = scandir($dir);
            foreach ($objects as $object) {
                if ($object != '.' && $object != '..') {
                    if (is_dir($dir . '/' . $object)) {
                        $this->recursive_delete($dir . '/' . $object);
                    } else {
                        unlink($dir . '/' . $object);
                    }
                }
            }
            rmdir($dir);
        }
    }

    /**
     * Create archive
     *
     * @param string $source_dir Source directory
     * @param string $archive_path Archive path
     */
    private function create_archive($source_dir, $archive_path) {
        $phar = new PharData($archive_path);
        $phar->buildFromDirectory($source_dir);
        $phar->compress(Phar::GZ);

        $tar_path = str_replace('.tar.gz', '.tar', $archive_path);
        if (file_exists($tar_path)) {
            unlink($tar_path);
        }
    }

    /**
     * Update component backup status
     *
     * @param string $component Component name
     * @param string $status Status
     * @param string|null $message Error message
     * @param array|null $backup Backup data
     */
    private function update_component_status($component, $status, $message = null, $backup = null) {
        $component_status = get_option('lox_wc_backup_component_status', array());

        $component_status[$component] = array(
            'last_backup' => current_time('mysql'),
            'status' => $status,
            'message' => $message,
            'uuid' => $backup ? $backup['uuid'] : null,
            'size' => $backup ? $backup['size_bytes'] : null,
        );

        update_option('lox_wc_backup_component_status', $component_status);
    }

    /**
     * Log message
     *
     * @param string $message Message
     * @param string $level Log level
     */
    private function log($message, $level = 'info') {
        if (defined('WP_DEBUG') && WP_DEBUG) {
            error_log(sprintf('[LOX WooCommerce Backup] [%s] %s', strtoupper($level), $message));
        }
    }

    /**
     * Format bytes to human readable
     *
     * @param int $bytes Bytes
     * @return string Formatted string
     */
    private function format_bytes($bytes) {
        $units = array('B', 'KB', 'MB', 'GB', 'TB');
        $bytes = max($bytes, 0);
        $pow = floor(($bytes ? log($bytes) : 0) / log(1024));
        $pow = min($pow, count($units) - 1);
        $bytes /= pow(1024, $pow);
        return round($bytes, 2) . ' ' . $units[$pow];
    }

    /**
     * Get WooCommerce-specific site metadata
     *
     * Extends WordPress site metadata with WooCommerce store information.
     *
     * @return array Site metadata including WooCommerce data
     */
    public function get_wc_site_metadata() {
        global $wp_version;

        // Base WordPress metadata
        $metadata = array(
            'site_url' => get_site_url(),
            'site_name' => get_bloginfo('name'),
            'wp_version' => $wp_version,
            'php_version' => phpversion(),
            'is_multisite' => is_multisite(),
            'theme' => wp_get_theme()->get('Name'),
            'plugin_count' => count(get_plugins()),
            'locale' => get_locale(),
            'source' => $this->source,
        );

        // WooCommerce-specific metadata
        if (function_exists('WC')) {
            $metadata['wc_version'] = WC()->version;
            $metadata['wc_db_version'] = get_option('woocommerce_db_version', '');

            // Product stats
            $metadata['product_count'] = (int) wp_count_posts('product')->publish;

            // Order stats (HPOS compatible)
            if (class_exists('Automattic\WooCommerce\Utilities\OrderUtil') &&
                \Automattic\WooCommerce\Utilities\OrderUtil::custom_orders_table_usage_is_enabled()) {
                global $wpdb;
                $metadata['order_count'] = (int) $wpdb->get_var("SELECT COUNT(*) FROM {$wpdb->prefix}wc_orders WHERE type = 'shop_order'");
            } else {
                $metadata['order_count'] = (int) wp_count_posts('shop_order')->{'wc-completed'} +
                                          (int) wp_count_posts('shop_order')->{'wc-processing'} +
                                          (int) wp_count_posts('shop_order')->{'wc-on-hold'};
            }

            // Customer count
            $metadata['customer_count'] = (int) count(get_users(array('role' => 'customer', 'fields' => 'ID')));

            // Store settings
            $metadata['currency'] = get_woocommerce_currency();
            $metadata['weight_unit'] = get_option('woocommerce_weight_unit', 'kg');
            $metadata['dimension_unit'] = get_option('woocommerce_dimension_unit', 'cm');

            // Payment gateways
            $gateways = WC()->payment_gateways()->get_available_payment_gateways();
            $metadata['payment_gateways'] = array_keys($gateways);

            // Shipping zones count
            if (class_exists('WC_Shipping_Zones')) {
                $metadata['shipping_zones_count'] = count(WC_Shipping_Zones::get_zones());
            }

            // Tax status
            $metadata['taxes_enabled'] = wc_tax_enabled();

            // Coupon count
            $metadata['coupon_count'] = (int) wp_count_posts('shop_coupon')->publish;
        }

        return $metadata;
    }

    /**
     * Run profile-based backup
     *
     * Creates a backup associated with a specific cloud profile.
     *
     * @param string $profile_uuid Profile UUID from LOX server
     * @param string $profile_name Profile name for display
     * @param array $components Components to backup (catalog, transactional, files, config)
     * @return array Result with success status and backup info
     */
    public function run_profile_backup($profile_uuid, $profile_name, $components = array('catalog')) {
        $start_time = time();
        $timestamp = date('Ymd_His');
        $site_name = sanitize_title(get_bloginfo('name'));

        // Create combined backup name
        $component_suffix = implode('-', $components);
        $backup_name = "{$this->source}-{$site_name}-{$profile_name}-{$component_suffix}-{$timestamp}";

        $this->log("Starting WooCommerce profile backup: {$backup_name}");

        try {
            // Create temp directory
            $temp_dir = $this->get_backup_dir_path() . '/temp_wc_profile_' . $timestamp;
            wp_mkdir_p($temp_dir);

            // Backup each component
            foreach ($components as $component) {
                if (!isset($this->components[$component])) {
                    continue;
                }

                $component_def = $this->components[$component];

                // Backup tables
                if (!empty($component_def['tables'])) {
                    $db_file = $this->backup_wc_tables($temp_dir, $component_def['tables']);
                    if (!is_wp_error($db_file)) {
                        $this->log("Database tables backed up for {$component}");
                    }
                }

                // Backup post types
                if (!empty($component_def['post_types'])) {
                    $this->backup_post_types($temp_dir, $component_def['post_types']);
                }

                // Backup taxonomies
                if (!empty($component_def['taxonomies'])) {
                    $this->backup_taxonomies($temp_dir, $component_def['taxonomies']);
                }

                // Export data
                if ($component === 'catalog') {
                    $this->export_catalog_data($temp_dir);
                } elseif ($component === 'transactional') {
                    $this->export_transactional_data($temp_dir);
                } elseif ($component === 'config') {
                    $this->export_config_data($temp_dir);
                }

                // Backup paths
                if (!empty($component_def['paths'])) {
                    foreach ($component_def['paths'] as $path) {
                        $full_path = WP_CONTENT_DIR . '/' . $path;
                        if (is_dir($full_path)) {
                            $this->backup_directory($full_path, $temp_dir, $path);
                        }
                    }
                }
            }

            // Create archive
            $archive_path = $this->get_backup_dir_path() . '/' . $backup_name . '.tar.gz';
            $this->create_archive($temp_dir, $archive_path);
            $this->log("Archive created: " . $this->format_bytes(filesize($archive_path)));

            // Get WooCommerce site metadata
            $extra_metadata = $this->get_wc_site_metadata();
            $extra_metadata['backup_components'] = $components;
            $extra_metadata['profile_name'] = $profile_name;

            // Upload to LOX with profile_uuid
            $api = new LOX_API();
            $result = $api->upload_backup($archive_path, array(
                'name' => $backup_name,
                'tags' => "{$this->source}," . implode(',', $components) . ",profile",
                'description' => sprintf(
                    'WooCommerce profile backup (%s) from %s',
                    $profile_name,
                    home_url()
                ),
                'source' => $this->source,
                'component' => implode(',', $components),
                'profile_uuid' => $profile_uuid,
                'extra_metadata' => $extra_metadata,
            ));

            if (is_wp_error($result)) {
                $this->cleanup($temp_dir, $archive_path);
                return array(
                    'success' => false,
                    'error' => $result->get_error_message(),
                );
            }

            // Wait for completion
            $backup = $api->wait_for_completion($result['uuid'], 1800);

            // Cleanup
            $this->cleanup($temp_dir, $archive_path);

            if (is_wp_error($backup)) {
                return array(
                    'success' => false,
                    'error' => $backup->get_error_message(),
                );
            }

            $duration = time() - $start_time;
            $this->log(sprintf('Profile backup completed in %d seconds. UUID: %s', $duration, $backup['uuid']));

            return array(
                'success' => true,
                'uuid' => $backup['uuid'],
                'size_bytes' => $backup['size_bytes'] ?? filesize($archive_path),
                'duration' => $duration,
            );

        } catch (Exception $e) {
            $this->log("Profile backup failed: " . $e->getMessage(), 'error');
            return array(
                'success' => false,
                'error' => $e->getMessage(),
            );
        }
    }
}
