<?php
/**
 * LOX Backup Module for PrestaShop
 *
 * @author    LOX Backup <support@backlox.com>
 * @copyright 2024 LOX Backup
 * @license   MIT
 */

if (!defined('_PS_VERSION_')) {
    exit;
}

require_once dirname(__FILE__) . '/classes/LoxApi.php';
require_once dirname(__FILE__) . '/classes/LoxBackupManager.php';

class LoxBackup extends Module
{
    /**
     * Module constructor
     */
    public function __construct()
    {
        $this->name = 'loxbackup';
        $this->tab = 'administration';
        $this->version = '1.3.0';
        $this->author = 'LOX Backup';
        $this->need_instance = 0;
        $this->ps_versions_compliancy = array('min' => '1.7.0.0', 'max' => _PS_VERSION_);
        $this->bootstrap = true;

        parent::__construct();

        $this->displayName = $this->l('LOX Backup');
        $this->description = $this->l('Automated cold storage backups for your PrestaShop store.');
        $this->confirmUninstall = $this->l('Are you sure you want to uninstall LOX Backup?');
    }

    /**
     * Install module
     */
    public function install()
    {
        if (!parent::install()) {
            return false;
        }

        // Create admin tab
        $tab = new Tab();
        $tab->active = 1;
        $tab->class_name = 'AdminLoxBackup';
        $tab->name = array();
        foreach (Language::getLanguages(true) as $lang) {
            $tab->name[$lang['id_lang']] = 'LOX Backup';
        }
        $tab->id_parent = (int) Tab::getIdFromClassName('AdminAdvancedParameters');
        $tab->module = $this->name;
        $tab->add();

        // Register cron hook
        $this->registerHook('actionCronJob');

        // Set default configuration
        Configuration::updateValue('LOX_BACKUP_API_KEY', '');
        Configuration::updateValue('LOX_BACKUP_API_URL', 'https://backlox.com/api');
        Configuration::updateValue('LOX_BACKUP_DATABASE', true);
        Configuration::updateValue('LOX_BACKUP_FILES', true);
        Configuration::updateValue('LOX_BACKUP_IMAGES', true);
        Configuration::updateValue('LOX_BACKUP_MODULES', false);
        Configuration::updateValue('LOX_BACKUP_THEMES', false);
        Configuration::updateValue('LOX_BACKUP_SCHEDULE', 'daily');
        Configuration::updateValue('LOX_BACKUP_RETENTION', 30);
        Configuration::updateValue('LOX_BACKUP_IMMUTABLE_DAYS', '');  // Empty = use server default
        Configuration::updateValue('LOX_BACKUP_TAGS', 'prestashop');
        Configuration::updateValue('LOX_BACKUP_LAST_RUN', '');
        Configuration::updateValue('LOX_BACKUP_LAST_STATUS', '');

        // Component-based backup schedules
        Configuration::updateValue('LOX_BACKUP_COMPONENT_SCHEDULES', json_encode(array(
            'catalog' => 'weekly',
            'transactional' => 'daily',
            'files' => 'weekly',
            'config' => 'daily',
        )));
        Configuration::updateValue('LOX_BACKUP_COMPONENT_STATUS', '{}');

        return true;
    }

    /**
     * Uninstall module
     */
    public function uninstall()
    {
        // Remove admin tab
        $id_tab = (int) Tab::getIdFromClassName('AdminLoxBackup');
        if ($id_tab) {
            $tab = new Tab($id_tab);
            $tab->delete();
        }

        // Remove configuration
        Configuration::deleteByName('LOX_BACKUP_API_KEY');
        Configuration::deleteByName('LOX_BACKUP_API_URL');
        Configuration::deleteByName('LOX_BACKUP_DATABASE');
        Configuration::deleteByName('LOX_BACKUP_FILES');
        Configuration::deleteByName('LOX_BACKUP_IMAGES');
        Configuration::deleteByName('LOX_BACKUP_MODULES');
        Configuration::deleteByName('LOX_BACKUP_THEMES');
        Configuration::deleteByName('LOX_BACKUP_SCHEDULE');
        Configuration::deleteByName('LOX_BACKUP_RETENTION');
        Configuration::deleteByName('LOX_BACKUP_IMMUTABLE_DAYS');
        Configuration::deleteByName('LOX_BACKUP_TAGS');
        Configuration::deleteByName('LOX_BACKUP_LAST_RUN');
        Configuration::deleteByName('LOX_BACKUP_LAST_STATUS');
        Configuration::deleteByName('LOX_BACKUP_COMPONENT_SCHEDULES');
        Configuration::deleteByName('LOX_BACKUP_COMPONENT_STATUS');

        return parent::uninstall();
    }

    /**
     * Module configuration page
     */
    public function getContent()
    {
        $output = '';

        // Process form submission
        if (Tools::isSubmit('submitLoxBackupSettings')) {
            $output .= $this->postProcess();
        }

        // Process test connection
        if (Tools::isSubmit('testLoxConnection')) {
            $output .= $this->testConnection();
        }

        // Process manual backup
        if (Tools::isSubmit('runLoxBackup')) {
            $output .= $this->runManualBackup();
        }

        return $output . $this->renderForm();
    }

    /**
     * Process form submission
     */
    protected function postProcess()
    {
        $api_key = Tools::getValue('LOX_BACKUP_API_KEY');
        $api_url = Tools::getValue('LOX_BACKUP_API_URL');

        Configuration::updateValue('LOX_BACKUP_API_KEY', $api_key);
        Configuration::updateValue('LOX_BACKUP_API_URL', $api_url);
        Configuration::updateValue('LOX_BACKUP_DATABASE', (bool) Tools::getValue('LOX_BACKUP_DATABASE'));
        Configuration::updateValue('LOX_BACKUP_FILES', (bool) Tools::getValue('LOX_BACKUP_FILES'));
        Configuration::updateValue('LOX_BACKUP_IMAGES', (bool) Tools::getValue('LOX_BACKUP_IMAGES'));
        Configuration::updateValue('LOX_BACKUP_MODULES', (bool) Tools::getValue('LOX_BACKUP_MODULES'));
        Configuration::updateValue('LOX_BACKUP_THEMES', (bool) Tools::getValue('LOX_BACKUP_THEMES'));
        Configuration::updateValue('LOX_BACKUP_SCHEDULE', Tools::getValue('LOX_BACKUP_SCHEDULE'));
        Configuration::updateValue('LOX_BACKUP_RETENTION', (int) Tools::getValue('LOX_BACKUP_RETENTION'));
        // Immutable days: empty = use server default, 0 = no immutability, >0 = immutable for N days
        $immutableDays = Tools::getValue('LOX_BACKUP_IMMUTABLE_DAYS');
        Configuration::updateValue('LOX_BACKUP_IMMUTABLE_DAYS', $immutableDays === '' ? '' : (int) $immutableDays);
        Configuration::updateValue('LOX_BACKUP_TAGS', Tools::getValue('LOX_BACKUP_TAGS'));

        return $this->displayConfirmation($this->l('Settings saved successfully.'));
    }

    /**
     * Test API connection
     */
    protected function testConnection()
    {
        $api = new LoxApi(
            Configuration::get('LOX_BACKUP_API_KEY'),
            Configuration::get('LOX_BACKUP_API_URL')
        );

        $result = $api->testConnection();

        if ($result['success']) {
            return $this->displayConfirmation(
                $this->l('Connection successful!') . ' ' .
                sprintf($this->l('Tenant: %s'), $result['data']['name'])
            );
        }

        return $this->displayError($this->l('Connection failed: ') . $result['error']);
    }

    /**
     * Run manual backup
     */
    protected function runManualBackup()
    {
        set_time_limit(3600);

        $manager = new LoxBackupManager();
        $result = $manager->runBackup();

        if ($result['success']) {
            return $this->displayConfirmation(
                $this->l('Backup completed successfully!') . '<br>' .
                sprintf($this->l('UUID: %s'), $result['uuid']) . '<br>' .
                sprintf($this->l('Size: %s'), $this->formatBytes($result['size_bytes']))
            );
        }

        return $this->displayError($this->l('Backup failed: ') . $result['error']);
    }

    /**
     * Render configuration form
     */
    protected function renderForm()
    {
        $helper = new HelperForm();
        $helper->show_toolbar = false;
        $helper->table = $this->table;
        $helper->module = $this;
        $helper->default_form_language = $this->context->language->id;
        $helper->allow_employee_form_lang = Configuration::get('PS_BO_ALLOW_EMPLOYEE_FORM_LANG', 0);
        $helper->identifier = $this->identifier;
        $helper->submit_action = 'submitLoxBackupSettings';
        $helper->currentIndex = $this->context->link->getAdminLink('AdminModules', false) .
            '&configure=' . $this->name . '&tab_module=' . $this->tab . '&module_name=' . $this->name;
        $helper->token = Tools::getAdminTokenLite('AdminModules');

        $helper->tpl_vars = array(
            'fields_value' => $this->getConfigFormValues(),
            'languages' => $this->context->controller->getLanguages(),
            'id_language' => $this->context->language->id,
        );

        return $this->renderStatusCard() . $helper->generateForm(array($this->getConfigForm()));
    }

    /**
     * Render status card
     */
    protected function renderStatusCard()
    {
        $last_run = Configuration::get('LOX_BACKUP_LAST_RUN');
        $last_status = Configuration::get('LOX_BACKUP_LAST_STATUS');

        $api = new LoxApi(
            Configuration::get('LOX_BACKUP_API_KEY'),
            Configuration::get('LOX_BACKUP_API_URL')
        );
        $quota = $api->getQuota();

        $storage_used = '--';
        $storage_quota = '--';
        $connection_status = 'error';

        if (!empty($quota['success']) && isset($quota['data'])) {
            $connection_status = 'success';
            $storage_used = isset($quota['data']['used_bytes']) ? $this->formatBytes($quota['data']['used_bytes']) : '--';
            if (!empty($quota['data']['unlimited'])) {
                $storage_quota = $this->l('Unlimited');
            } elseif (isset($quota['data']['quota_bytes'])) {
                $storage_quota = $this->formatBytes($quota['data']['quota_bytes']);
            }
        }

        $this->context->smarty->assign(array(
            'last_run' => $last_run ? date('Y-m-d H:i:s', strtotime($last_run)) : $this->l('Never'),
            'last_status' => $last_status ?: $this->l('N/A'),
            'storage_used' => $storage_used,
            'storage_quota' => $storage_quota,
            'connection_status' => $connection_status,
            'module_path' => $this->_path,
            'module_link' => $this->context->link->getAdminLink('AdminLoxBackup'),
        ));

        return $this->display(__FILE__, 'views/templates/admin/status_card.tpl');
    }

    /**
     * Get configuration form structure
     */
    protected function getConfigForm()
    {
        return array(
            'form' => array(
                'legend' => array(
                    'title' => $this->l('LOX Backup Settings'),
                    'icon' => 'icon-cogs',
                ),
                'input' => array(
                    array(
                        'type' => 'text',
                        'label' => $this->l('API Key'),
                        'name' => 'LOX_BACKUP_API_KEY',
                        'desc' => $this->l('Your LOX API key. Get it from backlox.com'),
                        'required' => true,
                        'class' => 'fixed-width-xxl',
                    ),
                    array(
                        'type' => 'text',
                        'label' => $this->l('API URL'),
                        'name' => 'LOX_BACKUP_API_URL',
                        'desc' => $this->l('For self-hosted instances. Leave default for cloud service.'),
                        'class' => 'fixed-width-xxl',
                    ),
                    array(
                        'type' => 'switch',
                        'label' => $this->l('Backup Database'),
                        'name' => 'LOX_BACKUP_DATABASE',
                        'is_bool' => true,
                        'values' => array(
                            array('id' => 'active_on', 'value' => 1, 'label' => $this->l('Yes')),
                            array('id' => 'active_off', 'value' => 0, 'label' => $this->l('No')),
                        ),
                    ),
                    array(
                        'type' => 'switch',
                        'label' => $this->l('Backup Files'),
                        'name' => 'LOX_BACKUP_FILES',
                        'desc' => $this->l('Include upload and download directories'),
                        'is_bool' => true,
                        'values' => array(
                            array('id' => 'active_on', 'value' => 1, 'label' => $this->l('Yes')),
                            array('id' => 'active_off', 'value' => 0, 'label' => $this->l('No')),
                        ),
                    ),
                    array(
                        'type' => 'switch',
                        'label' => $this->l('Backup Images'),
                        'name' => 'LOX_BACKUP_IMAGES',
                        'desc' => $this->l('Include product and category images'),
                        'is_bool' => true,
                        'values' => array(
                            array('id' => 'active_on', 'value' => 1, 'label' => $this->l('Yes')),
                            array('id' => 'active_off', 'value' => 0, 'label' => $this->l('No')),
                        ),
                    ),
                    array(
                        'type' => 'switch',
                        'label' => $this->l('Backup Modules'),
                        'name' => 'LOX_BACKUP_MODULES',
                        'is_bool' => true,
                        'values' => array(
                            array('id' => 'active_on', 'value' => 1, 'label' => $this->l('Yes')),
                            array('id' => 'active_off', 'value' => 0, 'label' => $this->l('No')),
                        ),
                    ),
                    array(
                        'type' => 'switch',
                        'label' => $this->l('Backup Themes'),
                        'name' => 'LOX_BACKUP_THEMES',
                        'is_bool' => true,
                        'values' => array(
                            array('id' => 'active_on', 'value' => 1, 'label' => $this->l('Yes')),
                            array('id' => 'active_off', 'value' => 0, 'label' => $this->l('No')),
                        ),
                    ),
                    array(
                        'type' => 'select',
                        'label' => $this->l('Schedule'),
                        'name' => 'LOX_BACKUP_SCHEDULE',
                        'options' => array(
                            'query' => array(
                                array('id' => 'hourly', 'name' => $this->l('Hourly')),
                                array('id' => 'daily', 'name' => $this->l('Daily')),
                                array('id' => 'weekly', 'name' => $this->l('Weekly')),
                                array('id' => 'monthly', 'name' => $this->l('Monthly')),
                                array('id' => 'disabled', 'name' => $this->l('Disabled')),
                            ),
                            'id' => 'id',
                            'name' => 'name',
                        ),
                    ),
                    array(
                        'type' => 'text',
                        'label' => $this->l('Retention Days'),
                        'name' => 'LOX_BACKUP_RETENTION',
                        'desc' => $this->l('Number of days to keep backups'),
                        'class' => 'fixed-width-sm',
                        'suffix' => $this->l('days'),
                    ),
                    array(
                        'type' => 'text',
                        'label' => $this->l('Immutable Period'),
                        'name' => 'LOX_BACKUP_IMMUTABLE_DAYS',
                        'desc' => $this->l('Days during which backups cannot be deleted (anti-ransomware). Leave empty for server default, 0 for no protection.'),
                        'class' => 'fixed-width-sm',
                        'suffix' => $this->l('days'),
                    ),
                    array(
                        'type' => 'text',
                        'label' => $this->l('Tags'),
                        'name' => 'LOX_BACKUP_TAGS',
                        'desc' => $this->l('Comma-separated tags for organizing backups'),
                        'class' => 'fixed-width-xl',
                    ),
                ),
                'buttons' => array(
                    array(
                        'title' => $this->l('Test Connection'),
                        'icon' => 'process-icon-refresh',
                        'name' => 'testLoxConnection',
                        'type' => 'submit',
                        'class' => 'btn btn-default pull-right',
                    ),
                    array(
                        'title' => $this->l('Backup Now'),
                        'icon' => 'process-icon-download',
                        'name' => 'runLoxBackup',
                        'type' => 'submit',
                        'class' => 'btn btn-primary pull-right',
                    ),
                ),
                'submit' => array(
                    'title' => $this->l('Save'),
                ),
            ),
        );
    }

    /**
     * Get configuration form values
     */
    protected function getConfigFormValues()
    {
        return array(
            'LOX_BACKUP_API_KEY' => Configuration::get('LOX_BACKUP_API_KEY'),
            'LOX_BACKUP_API_URL' => Configuration::get('LOX_BACKUP_API_URL'),
            'LOX_BACKUP_DATABASE' => Configuration::get('LOX_BACKUP_DATABASE'),
            'LOX_BACKUP_FILES' => Configuration::get('LOX_BACKUP_FILES'),
            'LOX_BACKUP_IMAGES' => Configuration::get('LOX_BACKUP_IMAGES'),
            'LOX_BACKUP_MODULES' => Configuration::get('LOX_BACKUP_MODULES'),
            'LOX_BACKUP_THEMES' => Configuration::get('LOX_BACKUP_THEMES'),
            'LOX_BACKUP_SCHEDULE' => Configuration::get('LOX_BACKUP_SCHEDULE'),
            'LOX_BACKUP_RETENTION' => Configuration::get('LOX_BACKUP_RETENTION'),
            'LOX_BACKUP_IMMUTABLE_DAYS' => Configuration::get('LOX_BACKUP_IMMUTABLE_DAYS'),
            'LOX_BACKUP_TAGS' => Configuration::get('LOX_BACKUP_TAGS'),
        );
    }

    /**
     * Cron job hook
     */
    public function hookActionCronJob()
    {
        // Run component-based backups
        $this->runComponentCronJobs();

        // Run legacy full backup if enabled
        $schedule = Configuration::get('LOX_BACKUP_SCHEDULE');

        if ($schedule === 'disabled') {
            return;
        }

        $last_run = Configuration::get('LOX_BACKUP_LAST_RUN');
        $should_run = false;

        if (empty($last_run)) {
            $should_run = true;
        } else {
            $last_time = strtotime($last_run);
            $now = time();
            $diff = $now - $last_time;

            switch ($schedule) {
                case 'hourly':
                    $should_run = $diff >= 3600;
                    break;
                case 'daily':
                    $should_run = $diff >= 86400;
                    break;
                case 'weekly':
                    $should_run = $diff >= 604800;
                    break;
                case 'monthly':
                    $should_run = $diff >= 2592000;
                    break;
            }
        }

        if ($should_run) {
            $manager = new LoxBackupManager();
            $manager->runBackup();
        }
    }

    /**
     * Run component-based cron jobs
     */
    protected function runComponentCronJobs()
    {
        $schedules = Configuration::get('LOX_BACKUP_COMPONENT_SCHEDULES');
        $schedules = $schedules ? json_decode($schedules, true) : array();

        if (empty($schedules)) {
            return;
        }

        $manager = new LoxBackupManager();
        $statuses = $manager->getComponentStatus();

        $intervals = array(
            'hourly' => 3600,
            'daily' => 86400,
            'weekly' => 604800,
            'monthly' => 2592000,
        );

        foreach ($schedules as $component => $schedule) {
            if ($schedule === 'disabled') {
                continue;
            }

            $shouldRun = false;
            $lastRun = isset($statuses[$component]['timestamp']) ? $statuses[$component]['timestamp'] : null;

            if (empty($lastRun)) {
                $shouldRun = true;
            } else {
                $lastTime = strtotime($lastRun);
                $diff = time() - $lastTime;
                $interval = isset($intervals[$schedule]) ? $intervals[$schedule] : 86400;
                $shouldRun = $diff >= $interval;
            }

            if ($shouldRun) {
                $manager->runComponentBackup($component);
            }
        }
    }

    /**
     * Get cron job frequency
     */
    public function getCronFrequency()
    {
        return array(
            'hour' => -1,
            'day' => -1,
            'month' => -1,
            'day_of_week' => -1,
        );
    }

    /**
     * Format bytes to human readable
     */
    protected function formatBytes($bytes)
    {
        $units = array('B', 'KB', 'MB', 'GB', 'TB');
        $bytes = max($bytes, 0);
        $pow = floor(($bytes ? log($bytes) : 0) / log(1024));
        $pow = min($pow, count($units) - 1);
        $bytes /= pow(1024, $pow);

        return round($bytes, 2) . ' ' . $units[$pow];
    }
}
