<?php
/**
 * LOX Backup Admin Controller
 */

if (!defined('_PS_VERSION_')) {
    exit;
}

require_once _PS_MODULE_DIR_ . 'loxbackup/classes/LoxApi.php';
require_once _PS_MODULE_DIR_ . 'loxbackup/classes/LoxBackupManager.php';

class AdminLoxBackupController extends ModuleAdminController
{
    public function __construct()
    {
        $this->bootstrap = true;
        $this->display = 'view';
        parent::__construct();

        $this->meta_title = $this->l('LOX Backup');
    }

    public function setMedia($isNewTheme = false)
    {
        parent::setMedia($isNewTheme);

        // Load custom CSS
        $this->addCSS(_PS_MODULE_DIR_ . 'loxbackup/views/css/admin.css');
    }

    public function initContent()
    {
        parent::initContent();

        $action = Tools::getValue('action', 'dashboard');

        switch ($action) {
            case 'backups':
                $this->renderBackupsList();
                break;
            case 'schedules':
                $this->renderSchedules();
                break;
            case 'profiles':
                $this->renderProfiles();
                break;
            case 'ajax_list':
                $this->ajaxListBackups();
                break;
            case 'ajax_download':
                $this->ajaxDownloadBackup();
                break;
            case 'ajax_restore':
                $this->ajaxRestoreBackup();
                break;
            case 'ajax_backup':
                $this->ajaxRunBackup();
                break;
            case 'ajax_custom_backup':
                $this->ajaxRunCustomBackup();
                break;
            case 'ajax_save_profile':
                $this->ajaxSaveProfile();
                break;
            case 'ajax_delete_profile':
                $this->ajaxDeleteProfile();
                break;
            case 'ajax_run_profile':
                $this->ajaxRunProfile();
                break;
            case 'ajax_get_remote_profiles':
                $this->ajaxGetRemoteProfiles();
                break;
            case 'ajax_get_profile_versions':
                $this->ajaxGetProfileVersions();
                break;
            case 'ajax_run_remote_profile':
                $this->ajaxRunRemoteProfile();
                break;
            default:
                $this->renderDashboard();
        }
    }

    protected function renderDashboard()
    {
        $api = new LoxApi(
            Configuration::get('LOX_BACKUP_API_KEY'),
            Configuration::get('LOX_BACKUP_API_URL')
        );

        $quota = $api->getQuota();
        $lastRun = Configuration::get('LOX_BACKUP_LAST_RUN');
        $lastStatus = Configuration::get('LOX_BACKUP_LAST_STATUS');

        // Calculate storage percentage
        $storagePercent = 0;
        if (!empty($quota['success']) && isset($quota['data']) && empty($quota['data']['unlimited'])
            && isset($quota['data']['quota_bytes']) && $quota['data']['quota_bytes'] > 0
            && isset($quota['data']['used_bytes'])) {
            $storagePercent = round(($quota['data']['used_bytes'] / $quota['data']['quota_bytes']) * 100);
        }

        // Get component status
        $manager = new LoxBackupManager();
        $componentStatus = $manager->getComponentStatus();

        // Get schedules
        $schedules = Configuration::get('LOX_BACKUP_COMPONENT_SCHEDULES');
        $schedules = $schedules ? json_decode($schedules, true) : [];

        // Component definitions
        $components = [
            'database' => ['label' => $this->l('Database'), 'description' => $this->l('Products, orders, customers, settings'), 'icon' => 'icon-database'],
            'files' => ['label' => $this->l('Files'), 'description' => $this->l('Upload and download directories'), 'icon' => 'icon-folder'],
            'images' => ['label' => $this->l('Images'), 'description' => $this->l('Product and category images'), 'icon' => 'icon-picture-o'],
            'modules' => ['label' => $this->l('Modules'), 'description' => $this->l('Installed modules'), 'icon' => 'icon-puzzle-piece'],
            'themes' => ['label' => $this->l('Themes'), 'description' => $this->l('Installed themes'), 'icon' => 'icon-paint-brush'],
        ];

        // Prepare storage display values
        $storageUsed = '--';
        $storageQuota = '--';
        $connectionOk = !empty($quota['success']) && isset($quota['data']);

        if ($connectionOk) {
            $storageUsed = isset($quota['data']['used_bytes']) ? $this->formatBytes($quota['data']['used_bytes']) : '--';
            if (!empty($quota['data']['unlimited'])) {
                $storageQuota = $this->l('Unlimited');
            } elseif (isset($quota['data']['quota_bytes'])) {
                $storageQuota = $this->formatBytes($quota['data']['quota_bytes']);
            }
        }

        $this->context->smarty->assign([
            'module_link' => $this->context->link->getAdminLink('AdminLoxBackup'),
            'configure_link' => $this->context->link->getAdminLink('AdminModules') . '&configure=loxbackup',
            'last_run' => $lastRun ? date('Y-m-d H:i:s', strtotime($lastRun)) : $this->l('Never'),
            'last_status' => $lastStatus ?: 'N/A',
            'storage_used' => $storageUsed,
            'storage_quota' => $storageQuota,
            'storage_percent' => $storagePercent,
            'connection_ok' => $connectionOk,
            'component_status' => $componentStatus,
            'schedules' => $schedules,
            'components' => $components,
        ]);

        $this->content = $this->context->smarty->fetch(
            _PS_MODULE_DIR_ . 'loxbackup/views/templates/admin/dashboard.tpl'
        );

        $this->context->smarty->assign('content', $this->content);
    }

    protected function renderBackupsList()
    {
        $this->context->smarty->assign([
            'module_link' => $this->context->link->getAdminLink('AdminLoxBackup'),
            'ajax_url' => $this->context->link->getAdminLink('AdminLoxBackup'),
        ]);

        $this->content = $this->context->smarty->fetch(
            _PS_MODULE_DIR_ . 'loxbackup/views/templates/admin/backups_list.tpl'
        );

        $this->context->smarty->assign('content', $this->content);
    }

    protected function renderSchedules()
    {
        if (Tools::isSubmit('submitSchedules')) {
            $this->saveSchedules();
        }

        $schedules = Configuration::get('LOX_BACKUP_COMPONENT_SCHEDULES');
        $schedules = $schedules ? json_decode($schedules, true) : [];

        $defaults = [
            'database' => ['enabled' => true, 'frequency' => 'daily', 'time' => '03:00'],
            'files' => ['enabled' => true, 'frequency' => 'weekly', 'time' => '04:00'],
            'images' => ['enabled' => false, 'frequency' => 'weekly', 'time' => '04:30'],
            'modules' => ['enabled' => false, 'frequency' => 'weekly', 'time' => '05:00'],
            'themes' => ['enabled' => false, 'frequency' => 'weekly', 'time' => '05:30'],
            'full' => ['enabled' => false, 'frequency' => 'weekly', 'time' => '02:00'],
        ];

        foreach ($defaults as $key => $default) {
            if (!isset($schedules[$key])) {
                $schedules[$key] = $default;
            } elseif (!is_array($schedules[$key])) {
                // Migrate old format
                $schedules[$key] = [
                    'enabled' => $schedules[$key] !== 'disabled',
                    'frequency' => $schedules[$key] !== 'disabled' ? $schedules[$key] : 'daily',
                    'time' => '03:00',
                ];
            }
        }

        $components = [
            'database' => ['label' => $this->l('Database'), 'description' => $this->l('Products, orders, customers, settings'), 'icon' => 'icon-database'],
            'files' => ['label' => $this->l('Files'), 'description' => $this->l('Upload and download directories'), 'icon' => 'icon-folder'],
            'images' => ['label' => $this->l('Images'), 'description' => $this->l('Product and category images'), 'icon' => 'icon-picture-o'],
            'modules' => ['label' => $this->l('Modules'), 'description' => $this->l('Installed modules'), 'icon' => 'icon-puzzle-piece'],
            'themes' => ['label' => $this->l('Themes'), 'description' => $this->l('Installed themes'), 'icon' => 'icon-paint-brush'],
            'full' => ['label' => $this->l('Full Backup'), 'description' => $this->l('Complete store backup'), 'icon' => 'icon-archive'],
        ];

        $frequencies = [
            'disabled' => $this->l('Disabled'),
            'hourly' => $this->l('Hourly'),
            'daily' => $this->l('Daily'),
            'weekly' => $this->l('Weekly'),
            'monthly' => $this->l('Monthly'),
        ];

        // Build cron URL
        $cronToken = Configuration::get('LOX_BACKUP_CRON_TOKEN');
        if (empty($cronToken)) {
            $cronToken = Tools::passwdGen(32);
            Configuration::updateValue('LOX_BACKUP_CRON_TOKEN', $cronToken);
        }
        $cronUrl = Tools::getShopDomainSsl(true) . '/modules/loxbackup/cron.php?token=' . $cronToken;

        $this->context->smarty->assign([
            'module_link' => $this->context->link->getAdminLink('AdminLoxBackup'),
            'schedules' => $schedules,
            'components' => $components,
            'frequencies' => $frequencies,
            'cron_url' => $cronUrl,
            'schedule_message' => !empty($this->confirmations) ? $this->confirmations[0] : null,
        ]);

        $this->content = $this->context->smarty->fetch(
            _PS_MODULE_DIR_ . 'loxbackup/views/templates/admin/schedules.tpl'
        );

        $this->context->smarty->assign('content', $this->content);
    }

    protected function saveSchedules()
    {
        $schedules = [];
        $components = ['database', 'files', 'images', 'modules', 'themes', 'full'];

        foreach ($components as $component) {
            $schedules[$component] = [
                'enabled' => (bool) Tools::getValue("schedule_{$component}_enabled"),
                'frequency' => Tools::getValue("schedule_{$component}_frequency", 'disabled'),
                'time' => Tools::getValue("schedule_{$component}_time", '03:00'),
            ];
        }

        Configuration::updateValue('LOX_BACKUP_COMPONENT_SCHEDULES', json_encode($schedules));

        $this->confirmations[] = $this->l('Schedules saved successfully.');
    }

    protected function ajaxListBackups()
    {
        header('Content-Type: application/json');

        $page = (int) Tools::getValue('page', 1);

        $api = new LoxApi(
            Configuration::get('LOX_BACKUP_API_KEY'),
            Configuration::get('LOX_BACKUP_API_URL')
        );

        $result = $api->listBackups([
            'page' => $page,
            'page_size' => 20,
            'source' => 'prestashop',
        ]);

        if ($result['success']) {
            echo json_encode(['success' => true, 'data' => $result['data']]);
        } else {
            echo json_encode(['success' => false, 'error' => $result['error']]);
        }
        exit;
    }

    protected function ajaxDownloadBackup()
    {
        header('Content-Type: application/json');

        $uuid = Tools::getValue('uuid');

        if (empty($uuid)) {
            echo json_encode(['success' => false, 'error' => 'Invalid UUID']);
            exit;
        }

        $api = new LoxApi(
            Configuration::get('LOX_BACKUP_API_KEY'),
            Configuration::get('LOX_BACKUP_API_URL')
        );

        $result = $api->getDownloadUrl($uuid);

        if ($result['success'] && !empty($result['data']['download_url'])) {
            echo json_encode(['success' => true, 'download_url' => $result['data']['download_url']]);
        } else {
            echo json_encode([
                'success' => false,
                'error' => $result['error'] ?? $this->l('Download is being prepared. Please try again.')
            ]);
        }
        exit;
    }

    protected function ajaxRestoreBackup()
    {
        header('Content-Type: application/json');

        $uuid = Tools::getValue('uuid');

        if (empty($uuid)) {
            echo json_encode(['success' => false, 'error' => 'Invalid UUID']);
            exit;
        }

        $api = new LoxApi(
            Configuration::get('LOX_BACKUP_API_KEY'),
            Configuration::get('LOX_BACKUP_API_URL')
        );

        $result = $api->requestRestore($uuid);

        if ($result['success']) {
            echo json_encode([
                'success' => true,
                'message' => $this->l('Restore requested. ') . ($result['data']['estimated_time'] ?? '')
            ]);
        } else {
            echo json_encode(['success' => false, 'error' => $result['error']]);
        }
        exit;
    }

    protected function ajaxRunBackup()
    {
        header('Content-Type: application/json');

        set_time_limit(3600);

        $manager = new LoxBackupManager();
        $result = $manager->runBackup();

        if ($result['success']) {
            echo json_encode([
                'success' => true,
                'message' => $this->l('Backup completed!'),
                'uuid' => $result['uuid'],
                'size' => $this->formatBytes($result['size_bytes'])
            ]);
        } else {
            echo json_encode(['success' => false, 'error' => $result['error']]);
        }
        exit;
    }

    protected function formatBytes($bytes)
    {
        $units = ['B', 'KB', 'MB', 'GB', 'TB'];
        $bytes = max($bytes, 0);
        $pow = floor(($bytes ? log($bytes) : 0) / log(1024));
        $pow = min($pow, count($units) - 1);
        $bytes /= pow(1024, $pow);

        return round($bytes, 2) . ' ' . $units[$pow];
    }

    /**
     * Render custom profiles page
     */
    protected function renderProfiles()
    {
        $profiles = Configuration::get('LOX_BACKUP_PROFILES');
        $profiles = $profiles ? json_decode($profiles, true) : [];

        $elements = [
            'database' => [
                'label' => $this->l('Database'),
                'description' => $this->l('All PrestaShop database tables'),
                'icon' => 'icon-database',
            ],
            'files' => [
                'label' => $this->l('Files'),
                'description' => $this->l('Upload and download directories'),
                'icon' => 'icon-folder',
            ],
            'images' => [
                'label' => $this->l('Images'),
                'description' => $this->l('Product and category images'),
                'icon' => 'icon-picture-o',
            ],
            'modules' => [
                'label' => $this->l('Modules'),
                'description' => $this->l('All installed modules'),
                'icon' => 'icon-puzzle-piece',
            ],
            'themes' => [
                'label' => $this->l('Themes'),
                'description' => $this->l('All installed themes'),
                'icon' => 'icon-paint-brush',
            ],
            'config' => [
                'label' => $this->l('Configuration'),
                'description' => $this->l('Store settings file'),
                'icon' => 'icon-cog',
            ],
        ];

        $frequencies = [
            'disabled' => $this->l('Disabled'),
            'hourly' => $this->l('Hourly'),
            'daily' => $this->l('Daily'),
            'weekly' => $this->l('Weekly'),
            'monthly' => $this->l('Monthly'),
        ];

        $this->context->smarty->assign([
            'module_link' => $this->context->link->getAdminLink('AdminLoxBackup'),
            'ajax_url' => $this->context->link->getAdminLink('AdminLoxBackup'),
            'profiles' => $profiles,
            'elements' => $elements,
            'frequencies' => $frequencies,
        ]);

        $this->content = $this->context->smarty->fetch(
            _PS_MODULE_DIR_ . 'loxbackup/views/templates/admin/profiles.tpl'
        );

        $this->context->smarty->assign('content', $this->content);
    }

    /**
     * AJAX: Run custom backup
     */
    protected function ajaxRunCustomBackup()
    {
        header('Content-Type: application/json');

        $elements = Tools::getValue('elements');
        if (!is_array($elements) || empty($elements)) {
            echo json_encode(['success' => false, 'error' => $this->l('No elements selected')]);
            exit;
        }

        set_time_limit(3600);

        $manager = new LoxBackupManager();
        $result = $manager->runCustomBackup($elements);

        if ($result['success']) {
            echo json_encode([
                'success' => true,
                'message' => $this->l('Custom backup completed!'),
                'uuid' => $result['uuid'],
                'size' => $this->formatBytes($result['size_bytes']),
            ]);
        } else {
            echo json_encode(['success' => false, 'error' => $result['error']]);
        }
        exit;
    }

    /**
     * AJAX: Save profile
     */
    protected function ajaxSaveProfile()
    {
        header('Content-Type: application/json');

        $profileId = Tools::getValue('profile_id');
        $name = Tools::getValue('name');
        $elements = Tools::getValue('elements');
        $schedule = Tools::getValue('schedule', 'disabled');
        $time = Tools::getValue('time', '03:00');
        $retentionDays = Tools::getValue('retention_days');
        $immutableDays = Tools::getValue('immutable_days');

        if (empty($name)) {
            echo json_encode(['success' => false, 'error' => $this->l('Profile name is required')]);
            exit;
        }

        if (!is_array($elements) || empty($elements)) {
            echo json_encode(['success' => false, 'error' => $this->l('At least one element is required')]);
            exit;
        }

        $profiles = Configuration::get('LOX_BACKUP_PROFILES');
        $profiles = $profiles ? json_decode($profiles, true) : [];

        // Generate ID if new
        if (empty($profileId)) {
            $profileId = 'profile_' . uniqid();
        }

        // Preserve existing stats
        $existing = isset($profiles[$profileId]) ? $profiles[$profileId] : [];
        $remoteUuid = isset($existing['remote_uuid']) ? $existing['remote_uuid'] : null;

        // Sync profile with server to get remote_uuid for versioning
        $api = new LoxApi(
            Configuration::get('LOX_BACKUP_API_KEY'),
            Configuration::get('LOX_BACKUP_API_URL')
        );

        $profileData = [
            'name' => $name,
            'component' => implode(',', $elements),
            'is_active' => $schedule !== 'disabled',
        ];

        if ($remoteUuid) {
            // Update existing remote profile
            $result = $api->updateProfile($remoteUuid, $profileData);
        } else {
            // Create new remote profile
            $result = $api->createProfile($name, $profileData);
            if ($result['success'] && isset($result['data']['uuid'])) {
                $remoteUuid = $result['data']['uuid'];
            }
        }

        $profiles[$profileId] = [
            'name' => $name,
            'elements' => $elements,
            'schedule' => $schedule,
            'time' => $time,
            'retention_days' => $retentionDays !== '' ? (int) $retentionDays : null,
            'immutable_days' => $immutableDays !== '' ? (int) $immutableDays : null,
            'last_run' => isset($existing['last_run']) ? $existing['last_run'] : null,
            'last_status' => isset($existing['last_status']) ? $existing['last_status'] : null,
            'remote_uuid' => $remoteUuid,
            'created' => isset($existing['created']) ? $existing['created'] : date('Y-m-d H:i:s'),
            'updated' => date('Y-m-d H:i:s'),
        ];

        Configuration::updateValue('LOX_BACKUP_PROFILES', json_encode($profiles));

        echo json_encode([
            'success' => true,
            'profile_id' => $profileId,
            'remote_uuid' => $remoteUuid,
            'message' => $this->l('Profile saved!'),
        ]);
        exit;
    }

    /**
     * AJAX: Delete profile
     */
    protected function ajaxDeleteProfile()
    {
        header('Content-Type: application/json');

        $profileId = Tools::getValue('profile_id');

        if (empty($profileId)) {
            echo json_encode(['success' => false, 'error' => $this->l('Invalid profile')]);
            exit;
        }

        $profiles = Configuration::get('LOX_BACKUP_PROFILES');
        $profiles = $profiles ? json_decode($profiles, true) : [];

        if (!isset($profiles[$profileId])) {
            echo json_encode(['success' => false, 'error' => $this->l('Profile not found')]);
            exit;
        }

        unset($profiles[$profileId]);
        Configuration::updateValue('LOX_BACKUP_PROFILES', json_encode($profiles));

        echo json_encode(['success' => true, 'message' => $this->l('Profile deleted')]);
        exit;
    }

    /**
     * AJAX: Run profile backup
     */
    protected function ajaxRunProfile()
    {
        header('Content-Type: application/json');

        $profileId = Tools::getValue('profile_id');

        if (empty($profileId)) {
            echo json_encode(['success' => false, 'error' => $this->l('Invalid profile')]);
            exit;
        }

        $profiles = Configuration::get('LOX_BACKUP_PROFILES');
        $profiles = $profiles ? json_decode($profiles, true) : [];

        if (!isset($profiles[$profileId])) {
            echo json_encode(['success' => false, 'error' => $this->l('Profile not found')]);
            exit;
        }

        $profile = $profiles[$profileId];
        set_time_limit(3600);

        // Determine frequency from schedule
        $frequency = isset($profile['schedule']) && $profile['schedule'] !== 'disabled'
            ? $profile['schedule']
            : 'daily';

        $manager = new LoxBackupManager();

        // If profile has remote_uuid, use runProfileBackup for versioning
        if (!empty($profile['remote_uuid'])) {
            $result = $manager->runProfileBackup(
                $profile['remote_uuid'],
                $profile['name'],
                $profile['elements'],
                $frequency
            );
        } else {
            // Use custom backup with frequency
            $result = $manager->runCustomBackup($profile['elements'], $profile['name'], $frequency);
        }

        // Update profile status
        if ($result['success']) {
            $profiles[$profileId]['last_run'] = date('Y-m-d H:i:s');
            $profiles[$profileId]['last_status'] = 'completed';
            $profiles[$profileId]['last_uuid'] = $result['uuid'];
            Configuration::updateValue('LOX_BACKUP_PROFILES', json_encode($profiles));

            echo json_encode([
                'success' => true,
                'message' => $this->l('Profile backup completed!'),
                'uuid' => $result['uuid'],
                'size' => $this->formatBytes($result['size_bytes']),
            ]);
        } else {
            $profiles[$profileId]['last_run'] = date('Y-m-d H:i:s');
            $profiles[$profileId]['last_status'] = 'failed';
            Configuration::updateValue('LOX_BACKUP_PROFILES', json_encode($profiles));

            echo json_encode(['success' => false, 'error' => $result['error']]);
        }
        exit;
    }

    /**
     * AJAX: Get remote profiles from LOX API
     */
    protected function ajaxGetRemoteProfiles()
    {
        header('Content-Type: application/json');

        $api = new LoxApi(
            Configuration::get('LOX_BACKUP_API_KEY'),
            Configuration::get('LOX_BACKUP_API_URL')
        );

        $result = $api->getProfiles();

        if ($result['success']) {
            echo json_encode([
                'success' => true,
                'profiles' => isset($result['data']['profiles']) ? $result['data']['profiles'] : $result['data'],
            ]);
        } else {
            echo json_encode(['success' => false, 'error' => $result['error']]);
        }
        exit;
    }

    /**
     * AJAX: Get profile versions (backups)
     */
    protected function ajaxGetProfileVersions()
    {
        header('Content-Type: application/json');

        $uuid = Tools::getValue('uuid');
        $page = (int) Tools::getValue('page', 1);

        if (empty($uuid)) {
            echo json_encode(['success' => false, 'error' => $this->l('Invalid profile UUID')]);
            exit;
        }

        $api = new LoxApi(
            Configuration::get('LOX_BACKUP_API_KEY'),
            Configuration::get('LOX_BACKUP_API_URL')
        );

        $result = $api->getProfileVersions($uuid, $page);

        if ($result['success']) {
            echo json_encode([
                'success' => true,
                'versions' => $result['data']['backups'],
                'total' => $result['data']['total'],
                'page' => $result['data']['page'],
                'per_page' => $result['data']['per_page'],
            ]);
        } else {
            echo json_encode(['success' => false, 'error' => $result['error']]);
        }
        exit;
    }

    /**
     * AJAX: Run backup for a remote profile
     */
    protected function ajaxRunRemoteProfile()
    {
        header('Content-Type: application/json');

        $profileUuid = Tools::getValue('profile_uuid');
        $profileName = Tools::getValue('profile_name', 'Profile Backup');
        $elements = Tools::getValue('elements');
        $frequency = Tools::getValue('frequency', 'daily');

        if (empty($profileUuid)) {
            echo json_encode(['success' => false, 'error' => $this->l('Invalid profile UUID')]);
            exit;
        }

        if (!is_array($elements) || empty($elements)) {
            // Default to database backup if no elements specified
            $elements = ['database'];
        }

        set_time_limit(3600);

        $manager = new LoxBackupManager();
        $result = $manager->runProfileBackup($profileUuid, $profileName, $elements, $frequency);

        if ($result['success']) {
            echo json_encode([
                'success' => true,
                'message' => $this->l('Profile backup completed!'),
                'uuid' => $result['uuid'],
                'size' => $this->formatBytes($result['size_bytes']),
                'profile_uuid' => $profileUuid,
            ]);
        } else {
            echo json_encode(['success' => false, 'error' => $result['error']]);
        }
        exit;
    }
}
