<?php
/**
 * LOX API Client for PrestaShop
 *
 * @author    LOX Backup <support@backlox.com>
 * @copyright 2024 LOX Backup
 * @license   MIT
 */

if (!defined('_PS_VERSION_')) {
    exit;
}

class LoxApi
{
    /**
     * @var string API key
     */
    private $apiKey;

    /**
     * @var string Base URL
     */
    private $baseUrl;

    /**
     * @var int Timeout in seconds
     */
    private $timeout = 300;

    /**
     * Constructor
     *
     * @param string $apiKey
     * @param string $baseUrl
     */
    public function __construct($apiKey, $baseUrl = 'https://backlox.com/api')
    {
        $this->apiKey = $apiKey;
        $this->baseUrl = rtrim($baseUrl, '/');
    }

    /**
     * Test API connection
     *
     * @return array
     */
    public function testConnection()
    {
        return $this->request('GET', '/v1/tenants/me');
    }

    /**
     * Get tenant quota
     *
     * @return array
     */
    public function getQuota()
    {
        return $this->request('GET', '/v1/tenants/usage');
    }

    /**
     * Upload backup file
     *
     * @param string $filePath Path to backup file
     * @param array $options Backup options
     * @return array
     */
    public function uploadBackup($filePath, $options = array())
    {
        if (!file_exists($filePath)) {
            return array('success' => false, 'error_code' => 'lox_upload_file_not_found', 'error' => 'Backup file not found');
        }

        $defaults = array(
            'name' => basename($filePath),
            'tags' => 'prestashop',
            'retention_days' => null,  // null = use frequency-based default from account settings
            'immutable_days' => null,  // null = use frequency-based default from account settings
            'frequency' => null,       // hourly, daily, weekly, monthly, yearly - determines retention policy
            'description' => 'PrestaShop backup from ' . Tools::getShopDomain(),
            'source' => 'prestashop',
            'component' => null,
            'source_identifier' => $this->getSiteIdentifier(),
            'profile_uuid' => null,  // Optional: assign to specific profile
            'extra_metadata' => null,  // Optional: site metadata
        );

        $options = array_merge($defaults, $options);

        // Auto-add site metadata if not provided
        if ($options['extra_metadata'] === null) {
            $options['extra_metadata'] = json_encode($this->getSiteMetadata());
        } elseif (is_array($options['extra_metadata'])) {
            $options['extra_metadata'] = json_encode($options['extra_metadata']);
        }

        $boundary = '----LoxBoundary' . uniqid();

        $body = '';

        // Add file
        $body .= "--{$boundary}\r\n";
        $body .= 'Content-Disposition: form-data; name="file"; filename="' . basename($filePath) . '"' . "\r\n";
        $body .= "Content-Type: application/gzip\r\n\r\n";
        $body .= file_get_contents($filePath);
        $body .= "\r\n";

        // Add form fields
        $fields = array('name', 'tags', 'retention_days', 'immutable_days', 'frequency', 'description', 'source', 'component', 'source_identifier', 'profile_uuid', 'extra_metadata');
        foreach ($fields as $field) {
            if (isset($options[$field]) && $options[$field] !== null) {
                $body .= "--{$boundary}\r\n";
                $body .= 'Content-Disposition: form-data; name="' . $field . '"' . "\r\n\r\n";
                $body .= $options[$field] . "\r\n";
            }
        }

        $body .= "--{$boundary}--\r\n";

        $ch = curl_init();
        curl_setopt_array($ch, array(
            CURLOPT_URL => $this->baseUrl . '/v1/backups',
            CURLOPT_POST => true,
            CURLOPT_POSTFIELDS => $body,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_TIMEOUT => $this->timeout,
            CURLOPT_HTTPHEADER => array(
                'X-API-Key: ' . $this->apiKey,
                'Content-Type: multipart/form-data; boundary=' . $boundary,
                'User-Agent: LOX-PrestaShop/1.0.0',
            ),
        ));

        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $error = curl_error($ch);
        curl_close($ch);

        if ($error) {
            return array('success' => false, 'error_code' => 'lox_conn_failed', 'error' => $error);
        }

        return $this->handleResponse($response, $httpCode);
    }

    /**
     * Get backup status
     *
     * @param string $uuid Backup UUID
     * @return array
     */
    public function getBackupStatus($uuid)
    {
        return $this->request('GET', '/v1/backups/' . $uuid);
    }

    /**
     * Get latest backups by component
     *
     * @param string $source Source identifier
     * @param string|null $sourceIdentifier Site identifier
     * @return array
     */
    public function getLatestBackups($source = 'prestashop', $sourceIdentifier = null)
    {
        $params = array('source' => $source);
        if ($sourceIdentifier === null) {
            $sourceIdentifier = $this->getSiteIdentifier();
        }
        if ($sourceIdentifier) {
            $params['source_identifier'] = $sourceIdentifier;
        }
        return $this->request('GET', '/v1/backups/latest', $params);
    }

    /**
     * Get backup profiles
     *
     * @param array $params Query parameters (source, source_identifier, is_active)
     * @return array
     */
    public function getProfiles($params = array())
    {
        $defaults = array(
            'source' => 'prestashop',
            'source_identifier' => $this->getSiteIdentifier(),
        );
        $params = array_merge($defaults, $params);
        return $this->request('GET', '/v1/backup-profiles/', $params);  // Trailing slash required
    }

    /**
     * Get a specific backup profile with its backups
     *
     * @param string $uuid Profile UUID
     * @param int $limit Number of backups to include
     * @return array
     */
    public function getProfile($uuid, $limit = 10)
    {
        return $this->request('GET', '/v1/backup-profiles/' . $uuid, array('limit' => $limit));
    }

    /**
     * Get all versions (backups) for a profile
     *
     * @param string $uuid Profile UUID
     * @param int $page Page number
     * @param int $perPage Items per page
     * @return array
     */
    public function getProfileVersions($uuid, $page = 1, $perPage = 20)
    {
        return $this->request('GET', '/v1/backup-profiles/' . $uuid . '/versions', array(
            'page' => $page,
            'per_page' => $perPage,
        ));
    }

    /**
     * Create a custom backup profile
     *
     * @param string $name Profile name
     * @param array $options Profile options
     * @return array
     */
    public function createProfile($name, $options = array())
    {
        $defaults = array(
            'name' => $name,
            'source' => 'prestashop',
            'source_identifier' => $this->getSiteIdentifier(),
            'is_custom' => true,
        );
        $data = array_merge($defaults, $options);
        return $this->request('POST', '/v1/backup-profiles/', $data);  // Trailing slash required
    }

    /**
     * Update a backup profile
     *
     * @param string $uuid Profile UUID
     * @param array $data Update data
     * @return array
     */
    public function updateProfile($uuid, $data)
    {
        return $this->request('PATCH', '/v1/backup-profiles/' . $uuid, $data);
    }

    /**
     * Get site metadata for backup profile
     *
     * @return array Site metadata
     */
    public function getSiteMetadata()
    {
        return array(
            'site_url' => Tools::getShopDomainSsl(true),
            'site_name' => Configuration::get('PS_SHOP_NAME'),
            'ps_version' => _PS_VERSION_,
            'php_version' => phpversion(),
            'is_multistore' => Shop::isFeatureActive(),
            'theme' => Context::getContext()->shop->theme_name,
            'module_count' => count(Module::getModulesInstalled()),
            'locale' => Context::getContext()->language->iso_code,
            'currency' => Context::getContext()->currency->iso_code,
        );
    }

    /**
     * Generate a unique identifier for this PrestaShop installation
     *
     * Uses a hash of the shop domain to create a consistent, privacy-friendly identifier.
     *
     * @return string Site identifier
     */
    public function getSiteIdentifier()
    {
        // Check if we have a stored identifier
        $storedId = Configuration::get('LOX_BACKUP_SITE_IDENTIFIER');
        if ($storedId) {
            return $storedId;
        }

        // Generate identifier based on shop domain
        $domain = Tools::getShopDomain();
        $shopId = (int) Context::getContext()->shop->id;

        // For multi-shop installations, include shop ID
        if (Shop::isFeatureActive() && $shopId > 1) {
            $identifier = $domain . '-shop' . $shopId;
        } else {
            $identifier = $domain;
        }

        // Add a short hash for uniqueness
        $hash = substr(md5($domain . _COOKIE_KEY_), 0, 8);
        $identifier = $identifier . '-' . $hash;

        // Store for consistency
        Configuration::updateValue('LOX_BACKUP_SITE_IDENTIFIER', $identifier);

        return $identifier;
    }

    /**
     * List backups
     *
     * @param array $params Query parameters
     * @return array
     */
    public function listBackups($params = array())
    {
        return $this->request('GET', '/v1/backups', $params);
    }

    /**
     * Request backup restore
     *
     * @param string $uuid Backup UUID
     * @return array
     */
    public function requestRestore($uuid)
    {
        return $this->request('POST', '/v1/backups/' . $uuid . '/restore', array(
            'priority' => 'normal'
        ));
    }

    /**
     * Get download URL for backup
     *
     * @param string $uuid Backup UUID
     * @return array
     */
    public function getDownloadUrl($uuid)
    {
        return $this->request('GET', '/v1/backups/' . $uuid . '/download');
    }

    /**
     * Wait for backup completion
     *
     * @param string $uuid Backup UUID
     * @param int $timeout Timeout in seconds
     * @return array
     */
    public function waitForCompletion($uuid, $timeout = 3600)
    {
        $startTime = time();
        $pollInterval = 5;

        while (true) {
            $backup = $this->getBackupStatus($uuid);

            if (!$backup['success']) {
                return $backup;
            }

            if ($backup['data']['status'] === 'completed') {
                return array(
                    'success' => true,
                    'data' => $backup['data'],
                );
            }

            if ($backup['data']['status'] === 'failed') {
                return array(
                    'success' => false,
                    'error_code' => 'lox_backup_failed',
                    'error' => 'Backup failed: ' . ($backup['data']['status_message'] ?? 'Unknown error'),
                );
            }

            if ($backup['data']['status'] === 'quarantine') {
                return array(
                    'success' => false,
                    'error_code' => 'lox_backup_quarantine',
                    'error' => 'Backup quarantined: Potential malware detected',
                );
            }

            if ((time() - $startTime) >= $timeout) {
                return array(
                    'success' => false,
                    'error_code' => 'lox_backup_timeout',
                    'error' => 'Timeout waiting for backup completion',
                );
            }

            sleep($pollInterval);
        }
    }

    /**
     * Make API request
     *
     * @param string $method HTTP method
     * @param string $endpoint API endpoint
     * @param array $params Request parameters
     * @return array
     */
    private function request($method, $endpoint, $params = array())
    {
        if (empty($this->apiKey)) {
            return array('success' => false, 'error_code' => 'lox_auth_missing_key', 'error' => 'API key not configured');
        }

        $url = $this->baseUrl . $endpoint;

        if ($method === 'GET' && !empty($params)) {
            $url .= '?' . http_build_query($params);
        }

        $ch = curl_init();
        curl_setopt_array($ch, array(
            CURLOPT_URL => $url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_TIMEOUT => 30,
            CURLOPT_HTTPHEADER => array(
                'X-API-Key: ' . $this->apiKey,
                'Content-Type: application/json',
                'User-Agent: LOX-PrestaShop/1.0.0',
            ),
        ));

        if ($method !== 'GET') {
            curl_setopt($ch, CURLOPT_CUSTOMREQUEST, $method);
            if (!empty($params)) {
                curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($params));
            }
        }

        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $error = curl_error($ch);
        curl_close($ch);

        if ($error) {
            return array('success' => false, 'error_code' => 'lox_conn_failed', 'error' => $error);
        }

        return $this->handleResponse($response, $httpCode);
    }

    /**
     * Handle API response
     *
     * @param string $response
     * @param int $httpCode
     * @return array
     */
    private function handleResponse($response, $httpCode)
    {
        $data = json_decode($response, true);

        if ($httpCode === 401) {
            return array('success' => false, 'error_code' => 'lox_auth_invalid_key', 'error' => 'Authentication failed. Check your API key.');
        }

        if ($httpCode === 403) {
            return array('success' => false, 'error_code' => 'lox_auth_forbidden', 'error' => 'Access denied');
        }

        if ($httpCode === 404) {
            return array('success' => false, 'error_code' => 'lox_backup_not_found', 'error' => 'Resource not found');
        }

        if ($httpCode === 413) {
            $message = isset($data['detail']) ? $data['detail'] : '';
            if (strpos(strtolower($message), 'quota') !== false) {
                return array('success' => false, 'error_code' => 'lox_upload_quota_exceeded', 'error' => 'Storage quota exceeded');
            }
            return array('success' => false, 'error_code' => 'lox_upload_file_too_large', 'error' => 'File too large');
        }

        if ($httpCode === 429) {
            return array('success' => false, 'error_code' => 'lox_rate_exceeded', 'error' => 'Rate limit exceeded. Please try again later.');
        }

        if ($httpCode >= 400) {
            $message = isset($data['detail']) ? $data['detail'] : 'API request failed';
            return array('success' => false, 'error_code' => 'lox_upload_failed', 'error' => $message);
        }

        return array('success' => true, 'data' => $data);
    }
}
