<?php

namespace Drupal\lox_backup\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Database\Connection;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Psr\Log\LoggerInterface;

/**
 * LOX Backup Manager service.
 */
class LoxBackupManager {

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * The file system service.
   *
   * @var \Drupal\Core\File\FileSystemInterface
   */
  protected $fileSystem;

  /**
   * The LOX API client.
   *
   * @var \Drupal\lox_backup\Service\LoxApi
   */
  protected $api;

  /**
   * The logger.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * Backup directory.
   *
   * @var string
   */
  protected $backupDir;

  /**
   * Source identifier.
   *
   * @var string
   */
  protected $source = 'drupal';

  /**
   * Component definitions.
   *
   * @var array
   */
  protected $components = [
    'content' => [
      'name' => 'Content',
      'description' => 'Nodes, taxonomy, comments, and media',
      'tables' => [
        'node',
        'node_field_data',
        'node_field_revision',
        'node_revision',
        'node__*',
        'node_revision__*',
        'taxonomy_term_data',
        'taxonomy_term_field_data',
        'taxonomy_term_field_revision',
        'taxonomy_term_revision',
        'taxonomy_term__*',
        'taxonomy_term_revision__*',
        'taxonomy_index',
        'comment',
        'comment_field_data',
        'comment__*',
        'comment_entity_statistics',
        'media',
        'media_field_data',
        'media_field_revision',
        'media_revision',
        'media__*',
        'media_revision__*',
        'file_managed',
        'file_usage',
        'path_alias',
        'path_alias_revision',
        'redirect',
        'block_content',
        'block_content_field_data',
        'block_content_field_revision',
        'block_content_revision',
        'menu_link_content',
        'menu_link_content_data',
        'menu_tree',
      ],
    ],
    'transactional' => [
      'name' => 'Users & Sessions',
      'description' => 'User accounts, roles, and session data',
      'tables' => [
        'users',
        'users_data',
        'users_field_data',
        'users_roles',
        'user__*',
        'sessions',
        'watchdog',
        'flood',
        'history',
        'batch',
        'queue',
        'cachetags',
        'cache_*',
        'key_value',
        'key_value_expire',
      ],
    ],
    'files' => [
      'name' => 'Media Files',
      'description' => 'Public and private uploaded files',
      'paths' => [
        'public://',
        'private://',
      ],
    ],
    'config' => [
      'name' => 'Configuration',
      'description' => 'Site configuration and active modules',
      'tables' => [
        'config',
        'router',
        'semaphore',
        'sequences',
      ],
      'export_config' => TRUE,
    ],
  ];

  /**
   * Constructs a LoxBackupManager object.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    Connection $database,
    FileSystemInterface $file_system,
    LoxApi $api,
    LoggerChannelFactoryInterface $logger_factory
  ) {
    $this->configFactory = $config_factory;
    $this->database = $database;
    $this->fileSystem = $file_system;
    $this->api = $api;
    $this->logger = $logger_factory->get('lox_backup');

    $this->backupDir = 'temporary://lox_backups';
    $this->fileSystem->prepareDirectory($this->backupDir, FileSystemInterface::CREATE_DIRECTORY);
  }

  /**
   * Run backup.
   *
   * @param array $options
   *   Options including 'component' for specific component backup.
   *
   * @return array
   *   Result array with backup information.
   */
  public function runBackup(array $options = []): array {
    // If a specific component is requested, run component backup.
    if (!empty($options['component']) && $options['component'] !== 'full') {
      return $this->runComponentBackup($options['component']);
    }

    // Run full backup.
    $timestamp = date('Ymd_His');
    $site_name = preg_replace('/[^a-z0-9]/', '-', strtolower(\Drupal::config('system.site')->get('name')));
    $backup_name = "drupal-{$site_name}-{$timestamp}";

    $this->logger->info('Starting backup: @name', ['@name' => $backup_name]);

    try {
      // Create temp directory.
      $temp_dir = $this->backupDir . '/temp_' . $timestamp;
      $this->fileSystem->prepareDirectory($temp_dir, FileSystemInterface::CREATE_DIRECTORY);

      $config = $this->configFactory->get('lox_backup.settings');

      // Backup database.
      if ($config->get('backup_database')) {
        $db_file = $this->backupDatabase($temp_dir);
        if ($db_file) {
          $this->logger->info('Database backup created');
        }
      }

      // Backup public files.
      if ($config->get('backup_files')) {
        $this->backupDirectory('public://', $temp_dir . '/files');
        $this->logger->info('Public files backup created');
      }

      // Backup private files.
      if ($config->get('backup_private')) {
        $private_path = \Drupal::service('stream_wrapper_manager')->getViaUri('private://');
        if ($private_path) {
          $this->backupDirectory('private://', $temp_dir . '/private');
          $this->logger->info('Private files backup created');
        }
      }

      // Backup configuration.
      if ($config->get('backup_config')) {
        $this->backupConfig($temp_dir);
        $this->logger->info('Configuration backup created');
      }

      // Create archive.
      $archive_path = $this->fileSystem->realpath($this->backupDir) . '/' . $backup_name . '.tar.gz';
      $this->createArchive($this->fileSystem->realpath($temp_dir), $archive_path);
      $this->logger->info('Archive created: @size', ['@size' => format_size(filesize($archive_path))]);

      // Upload to LOX.
      $this->logger->info('Uploading to LOX...');
      $schedule = $config->get('schedule') ?: 'daily';
      $result = $this->api->uploadBackup($archive_path, [
        'name' => $backup_name,
        'tags' => $config->get('tags') ?: 'drupal,automated',
        'retention_days' => NULL,  // Use frequency-based default
        'immutable_days' => $config->get('immutable_days'),
        'frequency' => $schedule !== 'disabled' ? $schedule : 'daily',
        'description' => sprintf(
          'Drupal backup from %s (%s)',
          \Drupal::request()->getHost(),
          \Drupal::config('system.site')->get('name')
        ),
      ]);

      if (!$result['success']) {
        $this->cleanup($temp_dir, $archive_path);
        $this->updateStatus('failed', $result['error']);
        return $result;
      }

      $this->logger->info('Upload complete, waiting for processing...');

      // Wait for completion.
      $backup = $this->api->waitForCompletion($result['data']['uuid'], 3600);

      // Cleanup.
      $this->cleanup($temp_dir, $archive_path);

      if (!$backup['success']) {
        $this->updateStatus('failed', $backup['error']);
        return $backup;
      }

      $this->logger->info('Backup completed. UUID: @uuid', ['@uuid' => $backup['data']['uuid']]);
      $this->updateStatus('completed', NULL, $backup['data']['uuid']);

      return [
        'success' => TRUE,
        'uuid' => $backup['data']['uuid'],
        'size_bytes' => $backup['data']['size_bytes'],
      ];
    }
    catch (\Exception $e) {
      $this->logger->error('Backup failed: @error', ['@error' => $e->getMessage()]);
      $this->updateStatus('failed', $e->getMessage());
      return ['success' => FALSE, 'error' => $e->getMessage()];
    }
  }

  /**
   * Backup database.
   *
   * @param string $temp_dir
   *   Temporary directory.
   *
   * @return string|null
   *   SQL file path or NULL on failure.
   */
  protected function backupDatabase(string $temp_dir): ?string {
    $sql_file = $temp_dir . '/database.sql';
    $handle = fopen($this->fileSystem->realpath($sql_file), 'w');

    if (!$handle) {
      return NULL;
    }

    // Write header.
    fwrite($handle, "-- LOX Drupal Backup\n");
    fwrite($handle, "-- Generated: " . date('Y-m-d H:i:s') . "\n");
    fwrite($handle, "-- Site: " . \Drupal::request()->getHost() . "\n\n");
    fwrite($handle, "SET NAMES utf8mb4;\n");
    fwrite($handle, "SET FOREIGN_KEY_CHECKS = 0;\n\n");

    // Get tables.
    $tables = $this->database->query("SHOW TABLES")->fetchCol();
    $prefix = $this->database->tablePrefix();

    foreach ($tables as $table_name) {
      // Only backup Drupal tables if prefix is set.
      if ($prefix && strpos($table_name, $prefix) !== 0) {
        continue;
      }

      // Table structure.
      $create = $this->database->query("SHOW CREATE TABLE `{$table_name}`")->fetchAssoc();
      fwrite($handle, "DROP TABLE IF EXISTS `{$table_name}`;\n");
      fwrite($handle, $create['Create Table'] . ";\n\n");

      // Table data.
      $rows = $this->database->query("SELECT * FROM `{$table_name}`")->fetchAll(\PDO::FETCH_ASSOC);

      if (!empty($rows)) {
        $columns = array_keys($rows[0]);
        $column_list = '`' . implode('`, `', $columns) . '`';

        foreach (array_chunk($rows, 100) as $chunk) {
          $values = [];
          foreach ($chunk as $row) {
            $escaped = array_map(function ($v) {
              return $v === NULL ? 'NULL' : "'" . addslashes($v) . "'";
            }, array_values($row));
            $values[] = '(' . implode(', ', $escaped) . ')';
          }
          fwrite($handle, "INSERT INTO `{$table_name}` ({$column_list}) VALUES\n" . implode(",\n", $values) . ";\n");
        }
        fwrite($handle, "\n");
      }
    }

    fwrite($handle, "SET FOREIGN_KEY_CHECKS = 1;\n");
    fclose($handle);

    return $sql_file;
  }

  /**
   * Backup a directory.
   *
   * @param string $source
   *   Source directory (can be stream wrapper).
   * @param string $dest
   *   Destination directory.
   */
  protected function backupDirectory(string $source, string $dest): void {
    $source_real = $this->fileSystem->realpath($source);
    if (!$source_real || !is_dir($source_real)) {
      return;
    }

    $this->fileSystem->prepareDirectory($dest, FileSystemInterface::CREATE_DIRECTORY);
    $dest_real = $this->fileSystem->realpath($dest);

    $this->recursiveCopy($source_real, $dest_real);
  }

  /**
   * Recursively copy directory.
   *
   * @param string $source
   *   Source path.
   * @param string $dest
   *   Destination path.
   */
  protected function recursiveCopy(string $source, string $dest): void {
    if (!is_dir($dest)) {
      mkdir($dest, 0755, TRUE);
    }

    $dir = opendir($source);
    while (($file = readdir($dir)) !== FALSE) {
      if ($file === '.' || $file === '..') {
        continue;
      }

      $src_path = $source . '/' . $file;
      $dest_path = $dest . '/' . $file;

      if (is_dir($src_path)) {
        $this->recursiveCopy($src_path, $dest_path);
      }
      else {
        copy($src_path, $dest_path);
      }
    }
    closedir($dir);
  }

  /**
   * Backup configuration.
   *
   * @param string $temp_dir
   *   Temporary directory.
   */
  protected function backupConfig(string $temp_dir): void {
    $config_dir = $temp_dir . '/config';
    $this->fileSystem->prepareDirectory($config_dir, FileSystemInterface::CREATE_DIRECTORY);
    $config_dir_real = $this->fileSystem->realpath($config_dir);

    // Export all configuration.
    $storage = \Drupal::service('config.storage');
    foreach ($storage->listAll() as $name) {
      $data = $storage->read($name);
      file_put_contents(
        $config_dir_real . '/' . $name . '.yml',
        \Symfony\Component\Yaml\Yaml::dump($data, 10, 2)
      );
    }
  }

  /**
   * Create tar.gz archive.
   *
   * @param string $source_dir
   *   Source directory.
   * @param string $archive_path
   *   Archive path.
   */
  protected function createArchive(string $source_dir, string $archive_path): void {
    $tar_path = str_replace('.tar.gz', '.tar', $archive_path);

    $phar = new \PharData($tar_path);
    $phar->buildFromDirectory($source_dir);
    $phar->compress(\Phar::GZ);

    if (file_exists($tar_path)) {
      unlink($tar_path);
    }
  }

  /**
   * Cleanup temporary files.
   *
   * @param string $temp_dir
   *   Temporary directory.
   * @param string|null $archive
   *   Archive file.
   */
  protected function cleanup(string $temp_dir, ?string $archive = NULL): void {
    $temp_dir_real = $this->fileSystem->realpath($temp_dir);
    if ($temp_dir_real && is_dir($temp_dir_real)) {
      $this->recursiveDelete($temp_dir_real);
    }

    if ($archive && file_exists($archive)) {
      unlink($archive);
    }
  }

  /**
   * Recursively delete directory.
   *
   * @param string $dir
   *   Directory path.
   */
  protected function recursiveDelete(string $dir): void {
    if (is_dir($dir)) {
      $objects = scandir($dir);
      foreach ($objects as $object) {
        if ($object !== '.' && $object !== '..') {
          if (is_dir($dir . '/' . $object)) {
            $this->recursiveDelete($dir . '/' . $object);
          }
          else {
            unlink($dir . '/' . $object);
          }
        }
      }
      rmdir($dir);
    }
  }

  /**
   * Update backup status.
   *
   * @param string $status
   *   Status string.
   * @param string|null $message
   *   Error message.
   * @param string|null $uuid
   *   Backup UUID.
   */
  protected function updateStatus(string $status, ?string $message = NULL, ?string $uuid = NULL): void {
    $config = $this->configFactory->getEditable('lox_backup.settings');
    $config->set('last_backup', date('Y-m-d\TH:i:s'));
    $config->set('last_backup_status', $status);
    if ($uuid) {
      $config->set('last_backup_uuid', $uuid);
    }
    $config->save();
  }

  /**
   * Get component definitions.
   *
   * @return array
   *   Component definitions.
   */
  public function getComponents(): array {
    return $this->components;
  }

  /**
   * Get source identifier.
   *
   * @return string
   *   Source identifier.
   */
  public function getSource(): string {
    return $this->source;
  }

  /**
   * Run component-specific backup.
   *
   * @param string $component
   *   Component name.
   *
   * @return array
   *   Result array with backup information.
   */
  public function runComponentBackup(string $component): array {
    if (!isset($this->components[$component])) {
      return ['success' => FALSE, 'error' => 'Unknown component: ' . $component];
    }

    $component_def = $this->components[$component];
    $timestamp = date('Ymd_His');
    $site_name = preg_replace('/[^a-z0-9]/', '-', strtolower(\Drupal::config('system.site')->get('name')));
    $backup_name = "drupal-{$site_name}-{$component}-{$timestamp}";

    $this->logger->info('Starting @component component backup: @name', [
      '@component' => $component,
      '@name' => $backup_name,
    ]);

    try {
      // Create temp directory.
      $temp_dir = $this->backupDir . '/temp_' . $component . '_' . $timestamp;
      $this->fileSystem->prepareDirectory($temp_dir, FileSystemInterface::CREATE_DIRECTORY);

      $config = $this->configFactory->get('lox_backup.settings');

      // Backup tables for this component.
      if (!empty($component_def['tables'])) {
        $this->backupComponentTables($temp_dir, $component_def['tables']);
      }

      // Backup paths for this component.
      if (!empty($component_def['paths'])) {
        foreach ($component_def['paths'] as $path) {
          $wrapper = \Drupal::service('stream_wrapper_manager')->getViaUri($path);
          if ($wrapper) {
            $dest_name = str_replace('://', '', $path);
            $this->backupDirectory($path, $temp_dir . '/' . $dest_name);
          }
        }
      }

      // Export configuration if needed.
      if (!empty($component_def['export_config'])) {
        $this->backupConfig($temp_dir);
      }

      // Export component-specific data.
      $this->exportComponentData($temp_dir, $component);

      // Create archive.
      $archive_path = $this->fileSystem->realpath($this->backupDir) . '/' . $backup_name . '.tar.gz';
      $this->createArchive($this->fileSystem->realpath($temp_dir), $archive_path);
      $this->logger->info('Archive created: @size', ['@size' => format_size(filesize($archive_path))]);

      // Upload to LOX.
      $this->logger->info('Uploading @component backup to LOX...', ['@component' => $component]);
      // Get frequency from component schedules
      $schedules = $config->get('component_schedules') ?: [];
      $frequency = $schedules[$component] ?? 'daily';
      $result = $this->api->uploadBackup($archive_path, [
        'name' => $backup_name,
        'tags' => ($config->get('tags') ?: 'drupal,automated') . ',' . $component,
        'retention_days' => NULL,  // Use frequency-based default
        'immutable_days' => $config->get('immutable_days'),
        'frequency' => $frequency !== 'disabled' ? $frequency : 'daily',
        'description' => sprintf(
          'Drupal %s backup from %s (%s)',
          $component_def['name'],
          \Drupal::request()->getHost(),
          \Drupal::config('system.site')->get('name')
        ),
        'source' => $this->source,
        'component' => $component,
      ]);

      if (!$result['success']) {
        $this->cleanup($temp_dir, $archive_path);
        $this->updateComponentStatus($component, 'failed', $result['error']);
        return $result;
      }

      $this->logger->info('Upload complete, waiting for processing...');

      // Wait for completion.
      $backup = $this->api->waitForCompletion($result['data']['uuid'], 3600);

      // Cleanup.
      $this->cleanup($temp_dir, $archive_path);

      if (!$backup['success']) {
        $this->updateComponentStatus($component, 'failed', $backup['error']);
        return $backup;
      }

      $this->logger->info('@component backup completed. UUID: @uuid', [
        '@component' => $component_def['name'],
        '@uuid' => $backup['data']['uuid'],
      ]);
      $this->updateComponentStatus($component, 'completed', NULL, $backup['data']['uuid']);

      return [
        'success' => TRUE,
        'uuid' => $backup['data']['uuid'],
        'size_bytes' => $backup['data']['size_bytes'],
        'component' => $component,
      ];
    }
    catch (\Exception $e) {
      $this->logger->error('Component backup failed: @error', ['@error' => $e->getMessage()]);
      $this->updateComponentStatus($component, 'failed', $e->getMessage());
      return ['success' => FALSE, 'error' => $e->getMessage()];
    }
  }

  /**
   * Backup specific tables for a component.
   *
   * @param string $temp_dir
   *   Temporary directory.
   * @param array $table_patterns
   *   Array of table name patterns (can include wildcards).
   */
  protected function backupComponentTables(string $temp_dir, array $table_patterns): void {
    $sql_file = $temp_dir . '/database.sql';
    $handle = fopen($this->fileSystem->realpath($sql_file), 'w');

    if (!$handle) {
      return;
    }

    $prefix = $this->database->tablePrefix();

    // Write header.
    fwrite($handle, "-- LOX Drupal Component Backup\n");
    fwrite($handle, "-- Generated: " . date('Y-m-d H:i:s') . "\n");
    fwrite($handle, "-- Site: " . \Drupal::request()->getHost() . "\n\n");
    fwrite($handle, "SET NAMES utf8mb4;\n");
    fwrite($handle, "SET FOREIGN_KEY_CHECKS = 0;\n\n");

    // Get all tables.
    $all_tables = $this->database->query("SHOW TABLES")->fetchCol();

    // Filter tables by patterns.
    $tables_to_backup = [];
    foreach ($all_tables as $table_name) {
      // Remove prefix for matching.
      $unprefixed = $prefix ? preg_replace('/^' . preg_quote($prefix, '/') . '/', '', $table_name) : $table_name;

      foreach ($table_patterns as $pattern) {
        // Convert wildcard pattern to regex.
        $regex = '/^' . str_replace('*', '.*', preg_quote($pattern, '/')) . '$/';
        if (preg_match($regex, $unprefixed)) {
          $tables_to_backup[] = $table_name;
          break;
        }
      }
    }

    foreach ($tables_to_backup as $table_name) {
      // Table structure.
      $create = $this->database->query("SHOW CREATE TABLE `{$table_name}`")->fetchAssoc();
      if (empty($create)) {
        continue;
      }

      fwrite($handle, "DROP TABLE IF EXISTS `{$table_name}`;\n");
      fwrite($handle, $create['Create Table'] . ";\n\n");

      // Table data.
      $rows = $this->database->query("SELECT * FROM `{$table_name}`")->fetchAll(\PDO::FETCH_ASSOC);

      if (!empty($rows)) {
        $columns = array_keys($rows[0]);
        $column_list = '`' . implode('`, `', $columns) . '`';

        foreach (array_chunk($rows, 100) as $chunk) {
          $values = [];
          foreach ($chunk as $row) {
            $escaped = array_map(function ($v) {
              return $v === NULL ? 'NULL' : "'" . addslashes($v) . "'";
            }, array_values($row));
            $values[] = '(' . implode(', ', $escaped) . ')';
          }
          fwrite($handle, "INSERT INTO `{$table_name}` ({$column_list}) VALUES\n" . implode(",\n", $values) . ";\n");
        }
        fwrite($handle, "\n");
      }
    }

    fwrite($handle, "SET FOREIGN_KEY_CHECKS = 1;\n");
    fclose($handle);

    $this->logger->info('Backed up @count tables', ['@count' => count($tables_to_backup)]);
  }

  /**
   * Export component-specific data to JSON.
   *
   * @param string $temp_dir
   *   Temporary directory.
   * @param string $component
   *   Component name.
   */
  protected function exportComponentData(string $temp_dir, string $component): void {
    $data_dir = $temp_dir . '/data';
    $this->fileSystem->prepareDirectory($data_dir, FileSystemInterface::CREATE_DIRECTORY);
    $data_dir_real = $this->fileSystem->realpath($data_dir);

    switch ($component) {
      case 'content':
        // Export content summary.
        $node_types = \Drupal::entityTypeManager()->getStorage('node_type')->loadMultiple();
        $content_stats = [];
        foreach ($node_types as $type) {
          $count = \Drupal::entityQuery('node')
            ->condition('type', $type->id())
            ->accessCheck(FALSE)
            ->count()
            ->execute();
          $content_stats[$type->id()] = [
            'label' => $type->label(),
            'count' => $count,
          ];
        }
        file_put_contents($data_dir_real . '/content_types.json', json_encode($content_stats, JSON_PRETTY_PRINT));

        // Export taxonomy vocabularies.
        $vocabularies = \Drupal::entityTypeManager()->getStorage('taxonomy_vocabulary')->loadMultiple();
        $vocab_data = [];
        foreach ($vocabularies as $vocab) {
          $count = \Drupal::entityQuery('taxonomy_term')
            ->condition('vid', $vocab->id())
            ->accessCheck(FALSE)
            ->count()
            ->execute();
          $vocab_data[$vocab->id()] = [
            'label' => $vocab->label(),
            'count' => $count,
          ];
        }
        file_put_contents($data_dir_real . '/vocabularies.json', json_encode($vocab_data, JSON_PRETTY_PRINT));
        break;

      case 'transactional':
        // Export user statistics.
        $user_count = \Drupal::entityQuery('user')
          ->accessCheck(FALSE)
          ->count()
          ->execute();
        $roles = \Drupal::entityTypeManager()->getStorage('user_role')->loadMultiple();
        $role_data = [];
        foreach ($roles as $role) {
          $role_data[$role->id()] = $role->label();
        }
        file_put_contents($data_dir_real . '/users_summary.json', json_encode([
          'total_users' => $user_count,
          'roles' => $role_data,
          'exported_at' => date('Y-m-d H:i:s'),
        ], JSON_PRETTY_PRINT));
        break;

      case 'config':
        // Export module list.
        $modules = \Drupal::moduleHandler()->getModuleList();
        $module_list = [];
        foreach ($modules as $name => $extension) {
          $module_list[] = $name;
        }
        file_put_contents($data_dir_real . '/enabled_modules.json', json_encode($module_list, JSON_PRETTY_PRINT));

        // Export theme list.
        $themes = \Drupal::service('theme_handler')->listInfo();
        $theme_list = [];
        foreach ($themes as $name => $theme) {
          $theme_list[$name] = [
            'name' => $theme->info['name'] ?? $name,
            'status' => $theme->status,
          ];
        }
        file_put_contents($data_dir_real . '/themes.json', json_encode($theme_list, JSON_PRETTY_PRINT));

        // Export site information.
        $site_config = \Drupal::config('system.site');
        file_put_contents($data_dir_real . '/site_info.json', json_encode([
          'name' => $site_config->get('name'),
          'slogan' => $site_config->get('slogan'),
          'mail' => $site_config->get('mail'),
          'front_page' => $site_config->get('page.front'),
        ], JSON_PRETTY_PRINT));
        break;

      case 'files':
        // Export file statistics.
        $file_count = \Drupal::entityQuery('file')
          ->accessCheck(FALSE)
          ->count()
          ->execute();
        file_put_contents($data_dir_real . '/files_summary.json', json_encode([
          'total_files' => $file_count,
          'exported_at' => date('Y-m-d H:i:s'),
        ], JSON_PRETTY_PRINT));
        break;
    }

    $this->logger->info('Exported @component data', ['@component' => $component]);
  }

  /**
   * Update component backup status.
   *
   * @param string $component
   *   Component name.
   * @param string $status
   *   Status string.
   * @param string|null $message
   *   Error message.
   * @param string|null $uuid
   *   Backup UUID.
   */
  protected function updateComponentStatus(string $component, string $status, ?string $message = NULL, ?string $uuid = NULL): void {
    $config = $this->configFactory->getEditable('lox_backup.settings');
    $statuses = $config->get('component_status') ?: [];

    $statuses[$component] = [
      'status' => $status,
      'timestamp' => date('Y-m-d\TH:i:s'),
      'message' => $message,
      'uuid' => $uuid,
    ];

    $config->set('component_status', $statuses);
    $config->save();
  }

  /**
   * Get component backup status.
   *
   * @param string|null $component
   *   Optional component name.
   *
   * @return array
   *   Component status(es).
   */
  public function getComponentStatus(?string $component = NULL): array {
    $config = $this->configFactory->get('lox_backup.settings');
    $statuses = $config->get('component_status') ?: [];

    if ($component) {
      return $statuses[$component] ?? [];
    }

    return $statuses;
  }

  /**
   * Run backup for a remote profile.
   *
   * @param string $profile_uuid
   *   Remote profile UUID.
   * @param string $profile_name
   *   Profile name for labeling.
   * @param array $components
   *   Components to backup (database, files, config).
   * @param string $frequency
   *   Backup frequency (daily, weekly, monthly).
   *
   * @return array
   *   Result array with backup information.
   */
  public function runProfileBackup(string $profile_uuid, string $profile_name, array $components = ['database'], string $frequency = 'daily'): array {
    if (empty($components)) {
      return ['success' => FALSE, 'error' => 'No backup components specified'];
    }

    $timestamp = date('Ymd_His');
    $site_name = preg_replace('/[^a-z0-9]/', '-', strtolower(\Drupal::config('system.site')->get('name')));
    $profile_slug = preg_replace('/[^a-z0-9]/', '-', strtolower($profile_name));
    $backup_name = "drupal-{$site_name}-{$profile_slug}-{$timestamp}";

    $this->logger->info('Starting profile backup: @name [Profile: @uuid]', [
      '@name' => $backup_name,
      '@uuid' => $profile_uuid,
    ]);

    try {
      // Create temp directory.
      $temp_dir = $this->backupDir . '/temp_profile_' . $timestamp;
      $this->fileSystem->prepareDirectory($temp_dir, FileSystemInterface::CREATE_DIRECTORY);

      $backed_up = [];

      // Backup database if requested.
      if (in_array('database', $components)) {
        $db_file = $this->backupDatabase($temp_dir);
        if ($db_file) {
          $backed_up[] = 'database';
          $this->logger->info('Database backup created');
        }
      }

      // Backup files if requested.
      if (in_array('files', $components)) {
        $this->backupDirectory('public://', $temp_dir . '/files');
        $backed_up[] = 'files';
        $this->logger->info('Public files backup created');
      }

      // Backup private files if requested.
      if (in_array('private', $components)) {
        $private_path = \Drupal::service('stream_wrapper_manager')->getViaUri('private://');
        if ($private_path) {
          $this->backupDirectory('private://', $temp_dir . '/private');
          $backed_up[] = 'private';
          $this->logger->info('Private files backup created');
        }
      }

      // Backup configuration if requested.
      if (in_array('config', $components)) {
        $this->backupConfig($temp_dir);
        $backed_up[] = 'config';
        $this->logger->info('Configuration backup created');
      }

      if (empty($backed_up)) {
        $this->cleanup($temp_dir);
        return ['success' => FALSE, 'error' => 'No files were backed up'];
      }

      // Create archive.
      $archive_path = $this->fileSystem->realpath($this->backupDir) . '/' . $backup_name . '.tar.gz';
      $this->createArchive($this->fileSystem->realpath($temp_dir), $archive_path);
      $this->logger->info('Archive created: @size', ['@size' => format_size(filesize($archive_path))]);

      // Build tags.
      $config = $this->configFactory->get('lox_backup.settings');
      $tags = array_merge(['drupal', 'profile'], $backed_up);
      $tags[] = 'profile:' . $profile_slug;

      // Upload to LOX with profile_uuid.
      $this->logger->info('Uploading profile backup to LOX...');
      $result = $this->api->uploadBackup($archive_path, [
        'name' => $backup_name,
        'tags' => implode(',', $tags),
        'retention_days' => NULL,  // Use frequency-based default
        'immutable_days' => $config->get('immutable_days'),
        'frequency' => $frequency,
        'description' => sprintf(
          'Drupal profile backup (%s) from %s (%s) - Profile: %s',
          implode(', ', $backed_up),
          \Drupal::request()->getHost(),
          \Drupal::config('system.site')->get('name'),
          $profile_name
        ),
        'source' => $this->source,
        'component' => 'profile',
        'profile_uuid' => $profile_uuid,
      ]);

      if (!$result['success']) {
        $this->cleanup($temp_dir, $archive_path);
        $this->updateStatus('failed', $result['error']);
        return $result;
      }

      $this->logger->info('Upload complete, waiting for processing...');

      // Wait for completion.
      $backup = $this->api->waitForCompletion($result['data']['uuid'], 3600);

      // Cleanup.
      $this->cleanup($temp_dir, $archive_path);

      if (!$backup['success']) {
        $this->updateStatus('failed', $backup['error']);
        return $backup;
      }

      $this->logger->info('Profile backup completed. UUID: @uuid', ['@uuid' => $backup['data']['uuid']]);
      $this->updateStatus('completed', NULL, $backup['data']['uuid']);

      return [
        'success' => TRUE,
        'uuid' => $backup['data']['uuid'],
        'size_bytes' => $backup['data']['size_bytes'],
        'profile_uuid' => $profile_uuid,
      ];
    }
    catch (\Exception $e) {
      $this->logger->error('Profile backup failed: @error', ['@error' => $e->getMessage()]);
      $this->updateStatus('failed', $e->getMessage());
      return ['success' => FALSE, 'error' => $e->getMessage()];
    }
  }

}
