<?php

namespace Drupal\lox_backup\Form;

use Drupal\Core\Entity\EntityForm;
use Drupal\Core\Form\FormStateInterface;

/**
 * Form handler for LOX Backup Schedule add/edit forms.
 */
class LoxBackupScheduleForm extends EntityForm {

  /**
   * {@inheritdoc}
   */
  public function form(array $form, FormStateInterface $form_state) {
    $form = parent::form($form, $form_state);

    /** @var \Drupal\lox_backup\Entity\LoxBackupSchedule $entity */
    $entity = $this->entity;

    $form['label'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Name'),
      '#maxlength' => 255,
      '#default_value' => $entity->label(),
      '#description' => $this->t('A descriptive name for this schedule.'),
      '#required' => TRUE,
    ];

    $form['id'] = [
      '#type' => 'machine_name',
      '#default_value' => $entity->id(),
      '#machine_name' => [
        'exists' => '\Drupal\lox_backup\Entity\LoxBackupSchedule::load',
      ],
      '#disabled' => !$entity->isNew(),
    ];

    $form['component'] = [
      '#type' => 'select',
      '#title' => $this->t('Component'),
      '#options' => [
        'full' => $this->t('Full Backup (All Components)'),
        'database' => $this->t('Database Only'),
        'files' => $this->t('Public Files Only'),
        'private' => $this->t('Private Files Only'),
        'config' => $this->t('Configuration Only'),
      ],
      '#default_value' => $entity->getComponent(),
      '#required' => TRUE,
      '#description' => $this->t('Select which component to backup.'),
    ];

    $form['interval'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Schedule'),
    ];

    $form['interval']['interval_number'] = [
      '#type' => 'number',
      '#title' => $this->t('Run every'),
      '#min' => 1,
      '#max' => 365,
      '#default_value' => $entity->getIntervalNumber(),
      '#required' => TRUE,
    ];

    $form['interval']['interval_type'] = [
      '#type' => 'select',
      '#title' => $this->t('Period'),
      '#title_display' => 'invisible',
      '#options' => [
        'hours' => $this->t('Hours'),
        'days' => $this->t('Days'),
        'weeks' => $this->t('Weeks'),
        'months' => $this->t('Months'),
      ],
      '#default_value' => $entity->getIntervalType(),
      '#required' => TRUE,
    ];

    $form['status'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enabled'),
      '#default_value' => $entity->status(),
      '#description' => $this->t('Enable or disable this backup schedule.'),
    ];

    $form['weight'] = [
      '#type' => 'weight',
      '#title' => $this->t('Weight'),
      '#default_value' => $entity->getWeight(),
      '#delta' => 50,
      '#description' => $this->t('Schedules with lower weights run first.'),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function save(array $form, FormStateInterface $form_state) {
    $entity = $this->entity;

    // Set values from form.
    $entity->setComponent($form_state->getValue('component'));
    $entity->setIntervalNumber((int) $form_state->getValue('interval_number'));
    $entity->setIntervalType($form_state->getValue('interval_type'));
    $entity->setWeight((int) $form_state->getValue('weight'));

    $status = $entity->save();

    if ($status === SAVED_NEW) {
      $this->messenger()->addStatus($this->t('Created the %label backup schedule.', [
        '%label' => $entity->label(),
      ]));
    }
    else {
      $this->messenger()->addStatus($this->t('Saved the %label backup schedule.', [
        '%label' => $entity->label(),
      ]));
    }

    $form_state->setRedirectUrl($entity->toUrl('collection'));
  }

}
