<?php

namespace Drupal\lox_backup\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Form for displaying and managing backups list.
 */
class LoxBackupListForm extends FormBase {

  /**
   * The LOX API service.
   *
   * @var \Drupal\lox_backup\Service\LoxApi
   */
  protected $loxApi;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    $instance = new static();
    $instance->loxApi = $container->get('lox_backup.api');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'lox_backup_list_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $page = $this->getRequest()->query->get('page', 0);
    $limit = 20;
    $offset = $page * $limit;

    // Fetch backups from API.
    $result = $this->loxApi->listBackups([
      'limit' => $limit,
      'offset' => $offset,
    ]);

    if (!$result['success']) {
      $this->messenger()->addError($this->t('Failed to load backups: @error', [
        '@error' => $result['error'],
      ]));
      return $form;
    }

    $backups = $result['data']['backups'] ?? $result['data'] ?? [];
    $total = $result['data']['total'] ?? count($backups);

    // Status badges.
    $status_labels = [
      'pending' => ['label' => $this->t('Pending'), 'class' => 'status-pending'],
      'uploading' => ['label' => $this->t('Uploading'), 'class' => 'status-uploading'],
      'validating' => ['label' => $this->t('Validating'), 'class' => 'status-validating'],
      'completed' => ['label' => $this->t('Completed'), 'class' => 'status-completed'],
      'failed' => ['label' => $this->t('Failed'), 'class' => 'status-failed'],
      'quarantine' => ['label' => $this->t('Quarantined'), 'class' => 'status-quarantine'],
    ];

    // Build table header.
    $header = [
      'date' => $this->t('Date'),
      'uuid' => $this->t('UUID'),
      'component' => $this->t('Component'),
      'size' => $this->t('Size'),
      'status' => $this->t('Status'),
      'operations' => $this->t('Operations'),
    ];

    $rows = [];
    foreach ($backups as $backup) {
      $uuid = $backup['id'] ?? $backup['uuid'] ?? '';
      $status = strtolower($backup['status'] ?? 'unknown');
      $status_info = $status_labels[$status] ?? ['label' => ucfirst($status), 'class' => ''];

      $operations = [];

      // Only show operations for completed backups.
      if ($status === 'completed') {
        $operations['restore'] = [
          '#type' => 'link',
          '#title' => $this->t('Restore'),
          '#url' => Url::fromRoute('lox_backup.restore', ['uuid' => $uuid]),
          '#attributes' => ['class' => ['button', 'button--small']],
        ];
        $operations['download'] = [
          '#type' => 'link',
          '#title' => $this->t('Download'),
          '#url' => Url::fromRoute('lox_backup.download', ['uuid' => $uuid]),
          '#attributes' => ['class' => ['button', 'button--small']],
        ];
      }
      elseif (in_array($status, ['pending', 'uploading', 'validating'])) {
        $operations['refresh'] = [
          '#type' => 'link',
          '#title' => $this->t('Refresh'),
          '#url' => Url::fromRoute('lox_backup.backups'),
          '#attributes' => ['class' => ['button', 'button--small']],
        ];
      }

      $rows[] = [
        'date' => [
          'data' => isset($backup['created_at'])
            ? \Drupal::service('date.formatter')->format(strtotime($backup['created_at']), 'short')
            : '-',
        ],
        'uuid' => [
          'data' => substr($uuid, 0, 8) . '...',
          'title' => $uuid,
        ],
        'component' => $backup['component'] ?? $backup['metadata']['component'] ?? '-',
        'size' => isset($backup['size']) ? format_size($backup['size']) : '-',
        'status' => [
          'data' => [
            '#type' => 'html_tag',
            '#tag' => 'span',
            '#value' => $status_info['label'],
            '#attributes' => ['class' => ['lox-status', $status_info['class']]],
          ],
        ],
        'operations' => [
          'data' => $operations,
        ],
      ];
    }

    $form['backups'] = [
      '#type' => 'table',
      '#header' => $header,
      '#rows' => $rows,
      '#empty' => $this->t('No backups found. Run a backup to see it here.'),
      '#attributes' => ['class' => ['lox-backups-table']],
    ];

    // Pager.
    if ($total > $limit) {
      $form['pager'] = [
        '#type' => 'container',
        '#attributes' => ['class' => ['lox-pager']],
      ];

      if ($page > 0) {
        $form['pager']['prev'] = [
          '#type' => 'link',
          '#title' => $this->t('« Previous'),
          '#url' => Url::fromRoute('lox_backup.backups', [], ['query' => ['page' => $page - 1]]),
          '#attributes' => ['class' => ['button']],
        ];
      }

      $form['pager']['info'] = [
        '#type' => 'html_tag',
        '#tag' => 'span',
        '#value' => $this->t('Page @current of @total', [
          '@current' => $page + 1,
          '@total' => ceil($total / $limit),
        ]),
        '#attributes' => ['class' => ['pager-info']],
      ];

      if (($offset + $limit) < $total) {
        $form['pager']['next'] = [
          '#type' => 'link',
          '#title' => $this->t('Next »'),
          '#url' => Url::fromRoute('lox_backup.backups', [], ['query' => ['page' => $page + 1]]),
          '#attributes' => ['class' => ['button']],
        ];
      }
    }

    // Add styles.
    $form['#attached']['library'][] = 'lox_backup/admin';

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    // No submit handling needed for list form.
  }

}
