<?php

namespace Drupal\lox_backup\Entity;

use Drupal\Core\Config\Entity\ConfigEntityBase;

/**
 * Defines the LOX Backup Schedule entity.
 *
 * @ConfigEntityType(
 *   id = "lox_backup_schedule",
 *   label = @Translation("LOX Backup Schedule"),
 *   label_collection = @Translation("LOX Backup Schedules"),
 *   label_singular = @Translation("backup schedule"),
 *   label_plural = @Translation("backup schedules"),
 *   handlers = {
 *     "list_builder" = "Drupal\lox_backup\LoxBackupScheduleListBuilder",
 *     "form" = {
 *       "add" = "Drupal\lox_backup\Form\LoxBackupScheduleForm",
 *       "edit" = "Drupal\lox_backup\Form\LoxBackupScheduleForm",
 *       "delete" = "Drupal\Core\Entity\EntityDeleteForm"
 *     },
 *     "route_provider" = {
 *       "html" = "Drupal\Core\Entity\Routing\AdminHtmlRouteProvider"
 *     }
 *   },
 *   config_prefix = "schedule",
 *   admin_permission = "administer lox backup",
 *   entity_keys = {
 *     "id" = "id",
 *     "label" = "label",
 *     "status" = "status"
 *   },
 *   config_export = {
 *     "id",
 *     "label",
 *     "status",
 *     "component",
 *     "interval_number",
 *     "interval_type",
 *     "weight"
 *   },
 *   links = {
 *     "add-form" = "/admin/config/system/lox-backup/schedules/add",
 *     "edit-form" = "/admin/config/system/lox-backup/schedules/{lox_backup_schedule}/edit",
 *     "delete-form" = "/admin/config/system/lox-backup/schedules/{lox_backup_schedule}/delete",
 *     "collection" = "/admin/config/system/lox-backup/schedules"
 *   }
 * )
 */
class LoxBackupSchedule extends ConfigEntityBase {

  /**
   * The schedule ID.
   *
   * @var string
   */
  protected $id;

  /**
   * The schedule label.
   *
   * @var string
   */
  protected $label;

  /**
   * Whether the schedule is enabled.
   *
   * @var bool
   */
  protected $status = TRUE;

  /**
   * The component to backup.
   *
   * @var string
   */
  protected $component = 'full';

  /**
   * The interval number.
   *
   * @var int
   */
  protected $interval_number = 1;

  /**
   * The interval type (hours, days, weeks, months).
   *
   * @var string
   */
  protected $interval_type = 'days';

  /**
   * Weight for ordering.
   *
   * @var int
   */
  protected $weight = 0;

  /**
   * Gets the component.
   *
   * @return string
   *   The component type.
   */
  public function getComponent(): string {
    return $this->component ?? 'full';
  }

  /**
   * Sets the component.
   *
   * @param string $component
   *   The component type.
   *
   * @return $this
   */
  public function setComponent(string $component): self {
    $this->component = $component;
    return $this;
  }

  /**
   * Gets the interval number.
   *
   * @return int
   *   The interval number.
   */
  public function getIntervalNumber(): int {
    return $this->interval_number ?? 1;
  }

  /**
   * Sets the interval number.
   *
   * @param int $number
   *   The interval number.
   *
   * @return $this
   */
  public function setIntervalNumber(int $number): self {
    $this->interval_number = $number;
    return $this;
  }

  /**
   * Gets the interval type.
   *
   * @return string
   *   The interval type.
   */
  public function getIntervalType(): string {
    return $this->interval_type ?? 'days';
  }

  /**
   * Sets the interval type.
   *
   * @param string $type
   *   The interval type.
   *
   * @return $this
   */
  public function setIntervalType(string $type): self {
    $this->interval_type = $type;
    return $this;
  }

  /**
   * Gets the cron interval in seconds.
   *
   * @return int
   *   The interval in seconds.
   */
  public function getIntervalSeconds(): int {
    $multipliers = [
      'hours' => 3600,
      'days' => 86400,
      'weeks' => 604800,
      'months' => 2592000, // 30 days
    ];

    $multiplier = $multipliers[$this->getIntervalType()] ?? 86400;
    return $this->getIntervalNumber() * $multiplier;
  }

  /**
   * Gets a human-readable interval string.
   *
   * @return string
   *   The interval string.
   */
  public function getIntervalString(): string {
    $number = $this->getIntervalNumber();
    $type = $this->getIntervalType();

    $labels = [
      'hours' => $number === 1 ? 'hour' : 'hours',
      'days' => $number === 1 ? 'day' : 'days',
      'weeks' => $number === 1 ? 'week' : 'weeks',
      'months' => $number === 1 ? 'month' : 'months',
    ];

    return $number . ' ' . ($labels[$type] ?? $type);
  }

  /**
   * Gets the weight.
   *
   * @return int
   *   The weight.
   */
  public function getWeight(): int {
    return $this->weight ?? 0;
  }

  /**
   * Sets the weight.
   *
   * @param int $weight
   *   The weight.
   *
   * @return $this
   */
  public function setWeight(int $weight): self {
    $this->weight = $weight;
    return $this;
  }

}
