<?php

namespace Drupal\lox_backup\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\File\FileSystemInterface;
use Drupal\lox_backup\Service\LoxBackupManager;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\BinaryFileResponse;
use Symfony\Component\HttpFoundation\ResponseHeaderBag;

/**
 * Controller for LOX PULL backup endpoints.
 *
 * Exposes secure endpoints for LOX to initiate backups remotely.
 * This enables the PULL model where LOX controls the backup schedule.
 */
class LoxPullController extends ControllerBase {

  protected $backupManager;
  protected $fileSystem;

  public function __construct(LoxBackupManager $backup_manager, FileSystemInterface $file_system) {
    $this->backupManager = $backup_manager;
    $this->fileSystem = $file_system;
  }

  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('lox_backup.manager'),
      $container->get('file_system')
    );
  }

  protected function verifyPullToken(Request $request): ?array {
    $token = $request->headers->get('X-LOX-Pull-Token');
    if (empty($token)) return ['error' => 'missing_token', 'message' => 'Missing X-LOX-Pull-Token header'];

    $config = $this->config('lox_backup.settings');
    $stored = $config->get('pull_token');
    $expiry = $config->get('pull_token_expiry');

    if (empty($stored)) return ['error' => 'not_configured', 'message' => 'PULL token not configured'];
    if ($expiry && time() > $expiry) return ['error' => 'token_expired', 'message' => 'Token expired'];
    if (!hash_equals($stored, $token)) return ['error' => 'invalid_token', 'message' => 'Invalid token'];

    return NULL;
  }

  protected function verifyApiKey(Request $request): ?array {
    $key = $request->headers->get('X-API-Key');
    if (empty($key)) return ['error' => 'missing_api_key', 'message' => 'Missing X-API-Key'];

    $stored = $this->config('lox_backup.settings')->get('api_key');
    if (empty($stored) || !hash_equals($stored, $key)) return ['error' => 'invalid_api_key', 'message' => 'Invalid API key'];

    return NULL;
  }

  public function backup(Request $request): JsonResponse {
    if ($error = $this->verifyPullToken($request)) return new JsonResponse($error, 401);

    $content = json_decode($request->getContent(), TRUE) ?: [];
    $type = $content['type'] ?? 'full';
    $frequency = $content['frequency'] ?? 'daily';

    try {
      $result = $this->runBackupForPull($type, $content, $frequency);
      if (isset($result['error'])) return new JsonResponse($result, 400);

      $token = $this->createDownloadToken($result['archive_path']);
      return new JsonResponse([
        'success' => TRUE,
        'backup_name' => $result['name'],
        'size_bytes' => filesize($result['archive_path']),
        'checksum' => hash_file('sha256', $result['archive_path']),
        'download_url' => '/lox/pull/download/' . $token,
        'expires_in' => 3600,
      ]);
    }
    catch (\Exception $e) {
      return new JsonResponse(['error' => 'backup_failed', 'message' => $e->getMessage()], 500);
    }
  }

  protected function runBackupForPull(string $type, array $opts, string $freq): array {
    $ts = date('Ymd_His');
    $site = preg_replace('/[^a-z0-9]/', '-', strtolower($this->config('system.site')->get('name')));
    $dir = 'temporary://lox_backups';
    $this->fileSystem->prepareDirectory($dir, FileSystemInterface::CREATE_DIRECTORY);

    $temp = $dir . '/temp_pull_' . $ts;
    $this->fileSystem->prepareDirectory($temp, FileSystemInterface::CREATE_DIRECTORY);
    $tempReal = $this->fileSystem->realpath($temp);

    $name = "drupal-{$site}";

    try {
      switch ($type) {
        case 'full':
          $name .= "-full-{$ts}";
          $this->collectFullBackup($tempReal);
          break;
        case 'component':
          $c = $opts['component'] ?? NULL;
          if (!$c) { $this->cleanupTemp($tempReal); return ['error' => 'missing_component']; }
          $name .= "-{$c}-{$ts}";
          $this->collectComponentBackup($tempReal, $c);
          break;
        case 'custom':
          $els = $opts['elements'] ?? [];
          if (!$els) { $this->cleanupTemp($tempReal); return ['error' => 'missing_elements']; }
          $slug = implode('-', array_map(fn($e) => preg_replace('/[^a-z0-9]/', '', strtolower($e)), $els));
          $name .= "-custom-{$slug}-{$ts}";
          $this->collectCustomBackup($tempReal, $els);
          break;
        default:
          $this->cleanupTemp($tempReal); return ['error' => 'invalid_type'];
      }

      if (count(scandir($tempReal)) <= 2) { $this->cleanupTemp($tempReal); return ['error' => 'backup_empty']; }

      $archive = $this->fileSystem->realpath($dir) . '/' . $name . '.tar.gz';
      $this->createArchive($tempReal, $archive);
      $this->cleanupTemp($tempReal);

      return ['name' => $name, 'archive_path' => $archive, 'type' => $type];
    }
    catch (\Exception $e) { $this->cleanupTemp($tempReal); throw $e; }
  }

  protected function collectFullBackup(string $dir): void {
    $cfg = $this->config('lox_backup.settings');
    if ($cfg->get('backup_database')) $this->backupDatabase($dir);
    if ($cfg->get('backup_files')) {
      $p = $this->fileSystem->realpath('public://');
      if ($p && is_dir($p)) $this->recursiveCopy($p, $dir . '/files');
    }
    if ($cfg->get('backup_private')) {
      $p = $this->fileSystem->realpath('private://');
      if ($p && is_dir($p)) $this->recursiveCopy($p, $dir . '/private');
    }
    if ($cfg->get('backup_config')) $this->backupConfig($dir);
  }

  protected function collectComponentBackup(string $dir, string $c): void {
    $comps = $this->backupManager->getComponents();
    if (!isset($comps[$c])) throw new \InvalidArgumentException('Invalid component');
    $def = $comps[$c];
    if (!empty($def['tables'])) $this->backupComponentTables($dir, $def['tables']);
    if (!empty($def['paths'])) foreach ($def['paths'] as $p) {
      $r = $this->fileSystem->realpath($p);
      if ($r && is_dir($r)) $this->recursiveCopy($r, $dir . '/' . str_replace('://', '', $p));
    }
    if (!empty($def['export_config'])) $this->backupConfig($dir);
  }

  protected function collectCustomBackup(string $dir, array $els): void {
    if (in_array('database', $els)) $this->backupDatabase($dir);
    if (in_array('files', $els)) {
      $p = $this->fileSystem->realpath('public://');
      if ($p && is_dir($p)) $this->recursiveCopy($p, $dir . '/files');
    }
    if (in_array('private', $els)) {
      $p = $this->fileSystem->realpath('private://');
      if ($p && is_dir($p)) $this->recursiveCopy($p, $dir . '/private');
    }
    if (in_array('config', $els)) $this->backupConfig($dir);
  }

  protected function backupDatabase(string $dir): void {
    $h = fopen($dir . '/database.sql', 'w');
    if (!$h) throw new \RuntimeException('Cannot create database file');
    $db = \Drupal::database();
    $pfx = $db->tablePrefix();

    fwrite($h, "-- LOX Drupal PULL Backup\n-- " . date('Y-m-d H:i:s') . "\n\nSET NAMES utf8mb4;\nSET FOREIGN_KEY_CHECKS = 0;\n\n");

    foreach ($db->query("SHOW TABLES")->fetchCol() as $t) {
      if ($pfx && strpos($t, $pfx) !== 0) continue;
      $c = $db->query("SHOW CREATE TABLE `{$t}`")->fetchAssoc();
      fwrite($h, "DROP TABLE IF EXISTS `{$t}`;\n{$c['Create Table']};\n\n");
      $rows = $db->query("SELECT * FROM `{$t}`")->fetchAll(\PDO::FETCH_ASSOC);
      if ($rows) {
        $cols = '`' . implode('`, `', array_keys($rows[0])) . '`';
        foreach (array_chunk($rows, 100) as $chunk) {
          $vals = array_map(fn($r) => '(' . implode(', ', array_map(fn($v) => $v === NULL ? 'NULL' : "'" . addslashes($v) . "'", $r)) . ')', $chunk);
          fwrite($h, "INSERT INTO `{$t}` ({$cols}) VALUES\n" . implode(",\n", $vals) . ";\n");
        }
      }
    }
    fwrite($h, "\nSET FOREIGN_KEY_CHECKS = 1;\n");
    fclose($h);
  }

  protected function backupComponentTables(string $dir, array $patterns): void {
    $h = fopen($dir . '/database.sql', 'w');
    if (!$h) throw new \RuntimeException('Cannot create database file');
    $db = \Drupal::database();
    $pfx = $db->tablePrefix();

    fwrite($h, "-- LOX Component Backup\n\nSET NAMES utf8mb4;\nSET FOREIGN_KEY_CHECKS = 0;\n\n");

    $all = $db->query("SHOW TABLES")->fetchCol();
    $tables = [];
    foreach ($all as $t) {
      $u = $pfx ? preg_replace('/^' . preg_quote($pfx, '/') . '/', '', $t) : $t;
      foreach ($patterns as $p) {
        if (preg_match('/^' . str_replace('*', '.*', preg_quote($p, '/')) . '$/', $u)) { $tables[] = $t; break; }
      }
    }

    foreach ($tables as $t) {
      $c = $db->query("SHOW CREATE TABLE `{$t}`")->fetchAssoc();
      if (!$c) continue;
      fwrite($h, "DROP TABLE IF EXISTS `{$t}`;\n{$c['Create Table']};\n\n");
      $rows = $db->query("SELECT * FROM `{$t}`")->fetchAll(\PDO::FETCH_ASSOC);
      if ($rows) {
        $cols = '`' . implode('`, `', array_keys($rows[0])) . '`';
        foreach (array_chunk($rows, 100) as $chunk) {
          $vals = array_map(fn($r) => '(' . implode(', ', array_map(fn($v) => $v === NULL ? 'NULL' : "'" . addslashes($v) . "'", $r)) . ')', $chunk);
          fwrite($h, "INSERT INTO `{$t}` ({$cols}) VALUES\n" . implode(",\n", $vals) . ";\n");
        }
      }
    }
    fwrite($h, "\nSET FOREIGN_KEY_CHECKS = 1;\n");
    fclose($h);
  }

  protected function backupConfig(string $dir): void {
    $cfg = $dir . '/config';
    if (!is_dir($cfg)) mkdir($cfg, 0755, TRUE);
    $storage = \Drupal::service('config.storage');
    foreach ($storage->listAll() as $n) {
      file_put_contents($cfg . '/' . $n . '.yml', \Symfony\Component\Yaml\Yaml::dump($storage->read($n), 10, 2));
    }
  }

  protected function recursiveCopy(string $s, string $d): void {
    if (!is_dir($d)) mkdir($d, 0755, TRUE);
    $dir = opendir($s);
    while (($f = readdir($dir)) !== FALSE) {
      if ($f === '.' || $f === '..') continue;
      is_dir("$s/$f") ? $this->recursiveCopy("$s/$f", "$d/$f") : copy("$s/$f", "$d/$f");
    }
    closedir($dir);
  }

  protected function createArchive(string $s, string $d): void {
    $tar = str_replace('.tar.gz', '.tar', $d);
    foreach ([$d, $tar] as $p) if (file_exists($p)) { try { \Phar::unlinkArchive($p); } catch (\Exception $e) { @unlink($p); } }
    $phar = new \PharData($tar);
    $phar->buildFromDirectory($s);
    $phar->compress(\Phar::GZ);
    unset($phar);
    if (file_exists($tar)) @unlink($tar);
  }

  protected function cleanupTemp(string $d): void {
    if (!is_dir($d)) return;
    foreach (scandir($d) as $f) {
      if ($f === '.' || $f === '..') continue;
      $p = "$d/$f";
      is_dir($p) ? $this->cleanupTemp($p) : unlink($p);
    }
    rmdir($d);
  }

  protected function createDownloadToken(string $path): string {
    $token = bin2hex(random_bytes(16));
    $dls = \Drupal::state()->get('lox_pull_downloads', []);
    $dls[$token] = ['path' => $path, 'expiry' => time() + 3600];
    \Drupal::state()->set('lox_pull_downloads', $dls);
    return $token;
  }

  public function download(string $token) {
    $dls = \Drupal::state()->get('lox_pull_downloads', []);
    if (!isset($dls[$token])) return new JsonResponse(['error' => 'invalid_token'], 404);

    $dl = $dls[$token];
    if (time() > $dl['expiry'] || !file_exists($dl['path'])) {
      unset($dls[$token]);
      \Drupal::state()->set('lox_pull_downloads', $dls);
      return new JsonResponse(['error' => 'expired'], 410);
    }

    unset($dls[$token]);
    \Drupal::state()->set('lox_pull_downloads', $dls);

    $resp = new BinaryFileResponse($dl['path']);
    $resp->setContentDisposition(ResponseHeaderBag::DISPOSITION_ATTACHMENT, basename($dl['path']));
    $resp->headers->set('Content-Type', 'application/gzip');
    $resp->headers->set('X-LOX-Checksum', hash_file('sha256', $dl['path']));
    $resp->deleteFileAfterSend(TRUE);
    return $resp;
  }

  public function status(Request $request): JsonResponse {
    if ($error = $this->verifyPullToken($request)) return new JsonResponse($error, 401);
    $cfg = $this->config('lox_backup.settings');
    $dbSize = 0;
    try { $dbSize = (int) \Drupal::database()->query("SELECT SUM(data_length + index_length) FROM information_schema.tables WHERE table_schema = DATABASE()")->fetchField(); } catch (\Exception $e) {}

    return new JsonResponse([
      'site_url' => \Drupal::request()->getSchemeAndHttpHost(),
      'site_name' => $this->config('system.site')->get('name'),
      'drupal_version' => \Drupal::VERSION,
      'module_version' => '1.3.0',
      'pull_enabled' => TRUE,
      'components' => [
        'database' => ['enabled' => (bool) $cfg->get('backup_database'), 'size_bytes' => $dbSize],
        'files' => ['enabled' => (bool) $cfg->get('backup_files')],
        'config' => ['enabled' => (bool) $cfg->get('backup_config')],
      ],
      'last_backup' => $cfg->get('last_backup'),
      'last_status' => $cfg->get('last_backup_status'),
    ]);
  }

  public function refreshToken(Request $request): JsonResponse {
    if ($error = $this->verifyApiKey($request)) return new JsonResponse($error, 401);
    $token = bin2hex(random_bytes(32));
    $expiry = time() + (365 * 86400);
    $cfg = $this->configFactory()->getEditable('lox_backup.settings');
    $cfg->set('pull_token', $token)->set('pull_token_expiry', $expiry)->save();
    return new JsonResponse(['success' => TRUE, 'token' => $token, 'expires_at' => date('c', $expiry)]);
  }

}
